# Notifications

This is a module for [Yii framework](http://www.yiiframework.com/) that provides:

* a configurable way to send various notifications, messages and tasks to a queue
* a basic widget to read such items from queue and display them to the user.

Messages could be passed directly as strings or created from two versions of a model, before and after updating its attributes. The created message could depend on:

* Model class
* Attributes modified
* Values before and after modification

As a side effect, this could be used to log all changes to the models, exactly like the [audittrail2](http://www.yiiframework.com/extension/audittrail2) extension.

Message filtering and creation logic is grouped into _channels_. Message recipients are selected by subscribing them to those channels.

A simple SQL queue implementation is provided if a MQ server is not available or not necessary.

## Components

A message is sent by calling Nfy::log method. Then, for each enabled channel:

* Channel's route class (NfyDbRoute) determines if message can be sent
* The message is filtered, creating the final string
* For each of channel's subscriptions, route determines if message can be delivered to subscribing user
* For each transport enabled for that subscription the message is delivered
* If the NfyDbRoute saved a messaged in a database queue, it could be displayed as a web notification by polling that queue

### Nfy

A simple helper class loading enabled channels, filtering message using levels and categories and handling processing to the channel's route.

This is an equivalent of a CLogger.

### NfyDbRoute

This class defines message delivery criteria, message formatting and implements transports. By default, it saves messages to a database queue.

It can be extended to implement other delivery methods, such as a Message Queue.

### WebNotifications

A widget displaying (html5) notifications from queues using ajax polling.

## Installation

Download and extract.

Enable module in configuration. Do it in both main and console configs, because some settings are used in migrations. See the configuration section how to specify users table name and its primary key type.

Import module classes from:
	'application.modules.nfy.components.Nfy',
	'application.modules.nfy.models.*',

Apply migrations:

~~~
./yiic migrate --migrationPath=nfy.migrations
~~~

Create some channels and subscribe users to them.

## Configuration

### Module parameters

By specifying the users model class name in the _userClass_ property proper table name and primary key column name will be used in migrations.

### Minimal setup

Provide an extended version of NfyDbRoute (called MyDbRoute in the example below) to implement custom message filtering/formatting.

~~~php
// initialize module's configuration, this should be done via migrations or CRUD
// create a channel with a basic filter and template
$channel = new NfyChannels;
$channel->name = 'test';
$channel->route_class = 'MyDbRoute';
$channel->message_template = 'Attribute changed from {old.attribute} to {new.attribute}';
$channel->save();
$subscription = new NfySubscriptions;
$subscription->user_id = Yii::app()->user->getId();
$subscription->channel_id = $channel->id;
$subscription->save();
~~~

## Usage examples

### Broadcasting

Send a message to every user.

First create a default channel and subscribe every user to it. Then just call Nfy::log().

### Notifying model changes

Monitor one model for changes and notify some users.

~~~php
// this could be your CActiveRecord model
class Test extends CActiveRecord {
	... // CActiveRecord requires some methods, this is skipped for keeping this short
	public static function match($data) {
		return $data['new']->attribute == 'value';
	}
	public function afterFind($event){
		$this->_old($this->getOwner()->getAttributes());
		return parent::afterFind($event);
	}
	public function afterSave($event) {
		$old = clone $this;
		$old->setAttributes($this->_old);
		Nfy::log(array('old'=>$old,'new'=>$this));
		return parent::afterSave($event);
	}
}

// trigger logging
$test = Test::model()->find();
$test->attribute = 'value';
$test->save();
// now the table nfy_queues contains a logged message
~~~

### Display notifications

Put anywhere in your layout or views or controller.

~~~
[php]
$this->widget('nfy.extensions.webNotifications.WebNotifications', array('url'=>$this->createUrl('/nfy/default/poll')));
~~~


## DB vs MQ

Implementing queues in a RDBMS could be considered as an anti-pattern. There are dedicated solutions to provide an asynchronous message queues. When it comes to notifications generated by an application, their delivery shouldn't block or cause exceptions in normal business logic execution. That's why a notification should be placed in a queue and another service should be reading from that queue and handling the delivery of notifications.

On the other hand, when messages are delivered only internally and there is no need or possibility to install another service, an RDBMS queue implementation could be sufficient and non disruptive.

An implementation of asynchronous notifications delivered through a Redis server could be found in the [vnotifier extension](http://www.yiiframework.com/extension/vnotifier).

## Changelog

### 0.5.5 - 2013-08-12

* Specify transport in the NfyChannels::subscribe() method.
* Fill value of registered_on when saving NfySubscriptions model.

### 0.5 - 2013-07-16

Backward compatibility breaking:

* Added routes and default NfyDbRoute implementation, this required refactoring the database schema, so remember to apply migrations when upgrading. The 'criteria_callback' column has been removed and replaced by the 'canSend()' method of the NfyDbRoute class.

Other changes:

* Saving the sending user id, so it can be used to filter recipients and customize message body.
* Added userClass parameter to the module to extract user table name and its primary key column name and type.
* Updated the documentation.

### 0.3 - 2013-07-13

* Initial release

## Todo

* Add push notifications example - add a server and replace ajax in webNotifications extension
* Add a NyMqRoute class
* Create a CRUD to manage channels and subscriptions and implement restrictions which channels are available for users to subscribe to.
* Provide a behavior similar to audit trail extension to plug in CActiveRecord
* Add two more transports to the NfyDbRoute: XMPP (jabber) and SMTP (email)
