<?php
/**
 * RezvanCaptchaAction class file.
 *
 * @author Nobuo Kihara <softark@gmail.com>
 * @link http://www.softark.net/
 * @copyright Copyright &copy; 2013 softark.net
 */

/**
 * RezvanCaptchaAction is an extension to CCaptchaAction.
 * 
 * RezvanCaptchaAction can render a CAPTCHA image with non alphabetical characters while CCaptchaAction is only for alphabets.
 *
 * RezvanCaptchaAction is used together with RezvanCaptcha and {@link CCaptchaValidator}.
 *
 * @author Nobuo Kihara <softark@gmail.com>
 */

class RezvanCaptchaAction extends CCaptchaAction
{
	/**
	 * @var integer the minimum length for randomly generated word. Defaults to 5.
	 */
	public $minLengthR = 5;

	/**
	 * @var integer the maximum length for randomly generated word. Defaults to 5.
	 */
	public $maxLengthR = 5;
	
	/**
	 * @var integer the offset between characters. Defaults to 2. You can adjust this property
	 * in order to decrease or increase the readability of the non alphabetical captcha.
	 **/
	public $offsetR = 2;

	/**
	 * @var boolean whether to use non alphabetical characters. Defaults to true.
	 */
	public $useRChars = true;

	/**
	 * @var string Non alphabetical font file. Defaults to BTC-Fa.ttf, a subset of
	 */
	public $fontFileR;
	
	/**
	 * @var boolean whether to render the captcha image with a fixed angle. Defaults to false.
	 * You may want to set this to true if you have trouble rendering your font.
	 */
	public $fixedAngle = false;
	
	/**
	 * @var string The string used for generating the random string of captcha.
	 * Defaults to a series of Persian characters. You may want to set your own.
	 */
	public $seeds;
	
	/**
	 * @var integer The minimum symobl height.
	 */
	public $font_size_min = 20;

	/**
	 * @var integer The maximum symobl height.
	 */
	public $font_size_max = 32;

	/**
	 * @var integer The noise level.
	 */
	public $noise = 2;
		
	/**
	 * Runs the action.
	 */
	public function run()
	{
		// Character type ... defaults to non alphabetical characters
		$session = Yii::app()->session;
		$session->open();
		$name = $this->getSessionKey();
		if ($session[$name . 'type'] !== null && $session[$name . 'type'] === 'abc')
		{
			$this->useRChars = false;
		}

		if (isset($_GET[self::REFRESH_GET_VAR]))  // AJAX request to refresh the code
		{
			$code = $this->getVerifyCode(true);
			echo CJSON::encode(array(
				'hash1'=>$this->generateValidationHash($code),
				'hash2'=>$this->generateValidationHash(strtolower($code)),
				// we add a random 'v' parameter so that FireFox can refresh the image
				// when src attribute of image tag is changed
				'url'=>$this->getController()->createUrl($this->getId(),array('v' => uniqid())),
			));
		}
		else
			$this->renderImage($this->getVerifyCode());

		Yii::app()->end();
	}

	/**
	 * Generates a hash code that can be used for client side validation.
	 * @param string $code the CAPTCHA code
	 * @return string a hash code generated from the CAPTCHA code
	 * 
	 * This generates a hash code from a UTF-8 string that is compatible with CCaptchaAction::generateValidationHash
	 */
	public function generateValidationHash($code)
	{
		$code = mb_convert_encoding($code, 'UCS-2', 'UTF-8');
		$hash = 0;
		for ( $i = strlen($code) / 2 - 1; $i >= 0; --$i)
		{
			$hash += ord($code[$i*2]) * 256 + ord($code[$i*2+1]);
		}
		return $hash;
	}
	
	/**
	 * Gets the verification code.
	 * @param string $regenerate whether the verification code should be regenerated.
	 * @return string the verification code.
	 */
	public function getVerifyCode($regenerate=false)
	{
		if($this->fixedVerifyCode !== null)
			return $this->fixedVerifyCode;

		$session = Yii::app()->session;
		$session->open();
		$name = $this->getSessionKey();
		if($session[$name] === null || $regenerate)
		{
			$session[$name] = $this->generateVerifyCode();
			$session[$name . 'type'] = $this->useRChars ? 'jchars' : 'abc';
			$session[$name . 'count'] = 1;
		}
		return $session[$name];
	}

	/**
	 * Generates a new verification code.
	 * @return string the generated verification code
	 */
	protected function generateVerifyCode()
	{
		// alphabets ?
		if (!$this->useRChars)
		{
			return parent::generateVerifyCode();
		}

		if($this->minLengthR < 3)
			$this->minLengthR = 3;
		if($this->maxLengthR > 20)
			$this->maxLengthR = 20;
		if($this->minLengthR > $this->maxLengthR)
			$this->maxLengthR = $this->minLengthR;
		$length = mt_rand($this->minLengthR, $this->maxLengthR);

		$letters = isset($this->seeds) ? $this->seeds : '2345689';
		$len = mb_strlen($letters, 'UTF-8');

		$code = '';
		for($i = 0; $i < $length; ++$i)
		{
			$code .= mb_substr($letters, mt_rand(0, $len - 1), 1, 'UTF-8');
		}

		return $code;
	}

	/**
	 * Renders the CAPTCHA image based on the code.
	 * @param string $code the verification code
	 * @return string image content
	 */
	protected function renderImage($code)
	{
		// alphabets ?
		if (!$this->useRChars)
		{
			parent::renderImage($code);
			return;
		}

		// font defaults to BTC-Fa.ttf
		if($this->fontFileR === null)
			$this->fontFileR = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'BTC-Fa.ttf';

		$encoding = 'UTF-8';
				
		if($this->backend===null && CCaptcha::checkRequirements('imagick') || $this->backend==='imagick')
			$this->renderImageImagickJ($code, $encoding);
		else if($this->backend===null && CCaptcha::checkRequirements('gd') || $this->backend==='gd')
			$this->renderImageGDJ($code, $encoding);
	}
	
	/**
	 * Renders the CAPTCHA image based on the code using GD.
	 * @param string $code the verification code
	 * @param string $encoding the encoding of the verification code
	 * @return string image content
	 */
	protected function renderImageGDJ($code, $encoding)
	{
		$image = imagecreatetruecolor($this->width,$this->height);

		$backColor = imagecolorallocate($image,
				(int)($this->backColor % 0x1000000 / 0x10000),
				(int)($this->backColor % 0x10000 / 0x100),
				$this->backColor % 0x100);
		imagefilledrectangle($image,0,0,$this->width,$this->height,$backColor);
		imagecolordeallocate($image,$backColor);

		if($this->transparent)
			imagecolortransparent($image,$backColor);

		$foreColor = imagecolorallocate($image,
				(int)($this->foreColor % 0x1000000 / 0x10000),
				(int)($this->foreColor % 0x10000 / 0x100),
				$this->foreColor % 0x100);

		$length = mb_strlen($code, $encoding);
		$box = imagettfbbox(30,0,$this->fontFileR,$code);
		$w = $box[4] - $box[0] + $this->offsetR * ($length - 1);
		$h = $box[1] - $box[5];
		if ($h <= 0)
		{
			$h = $w / $length;
		}
		$scale = min(($this->width - $this->padding * 2) / $w, ($this->height - $this->padding * 2) / $h);
		$x = 8;
		// font size and angle
		$fontSize = (int)(30 * $scale * 0.90);
		$angle = 0;
		// base line
		$ybottom = $this->height - $this->padding * 4;
		$ytop = (int)($h * $scale * 0.95) + $this->padding * 4;
		if ($ytop > $ybottom)
		{
			$ytop = $ybottom;
		}
		for($i = 0; $i < $length; ++$i)
		{
			$letter = mb_substr( $code, $i, 1, $encoding);
			$y = mt_rand($ytop, $ybottom);
			if (!$this->fixedAngle)
				$angle = mt_rand(-15, 15);
			$box = imagettftext($image,$fontSize,$angle,$x,$y,$foreColor,$this->fontFileR,$letter);
			$x = $box[2] + $this->offsetR;
		}

		imagecolordeallocate($image,$foreColor);

		header('Pragma: public');
		header('Expires: 0');
		header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
		header('Content-Transfer-Encoding: binary');
		header("Content-type: image/png");
		imagepng($image);
		imagedestroy($image);
	}

	/**
	 * Renders the CAPTCHA image based on the code using Imagick.
	 * @param string $code the verification code
	 * @param string $encoding the encoding of the verification code
	 * @return string image content
	 */
	protected function renderImageImagickR($code, $encoding)
	{
		$backColor = new ImagickPixel('#'.dechex($this->backColor));
		$foreColor = new ImagickPixel('#'.dechex($this->foreColor));

		$image = new Imagick();
		$image->newImage($this->width, $this->height, $backColor);

		$draw = new ImagickDraw();
		$draw->setFont($this->fontFileR);
		$draw->setFontSize(30);
		$fontMetrics=$image->queryFontMetrics($draw, $code);

		$length = mb_strlen($code, $encoding);
		$w = (int)($fontMetrics['textWidth']) + $this->offsetR * ($length-1);
		$h = (int)($fontMetrics['textHeight']);
		$scale = min(($this->width - $this->padding*2) / $w, ($this->height - $this->padding*2) / $h);
		$x=8;
		// font size and angle
		$fontSize = (int)(30 * $scale * 0.90);
		$angle = 0;
		// base line
		$ybottom = $this->height - $this->padding * 4;
		$ytop = (int)($h * $scale * 0.95) + $this->padding * 4;
		if ($ytop > $ybottom)
		{
			$ytop = $ybottom;
		}
		for($i = 0; $i < $length; ++$i)
		{
			$letter = mb_substr( $code, $i, 1, $encoding);
			$y = mt_rand($ytop, $ybottom);
			if (!$this->fixedAngle)
				$angle = mt_rand(-15, 15);
			$draw = new ImagickDraw();
			$draw->setFont($this->fontFileR);
			$draw->setFontSize($fontSize);
			$draw->setFillColor($foreColor);
			$image->annotateImage($draw, $x, $y, $angle, $letter);
			$fontMetrics = $image->queryFontMetrics($draw, $letter);
			$x += (int)($fontMetrics['textWidth']) + $this->offsetR;
		}

		header('Pragma: public');
		header('Expires: 0');
		header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
		header('Content-Transfer-Encoding: binary');
		header("Content-type: image/png");
		$image->setImageFormat('png');
		echo $image;
	}
}