<?php

/**
 * User.php
 *
 * Note:
 * Account, Person, Address are implemented as embedded documents.
 * They are different entities.
 * Maybe in future modules an external application needs an account for import/export,
 * Address is more a location which can be shared (company, family, event location ...)
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class User extends Page {
    public $roles;
	public $lastLogin;
	public $lastOnline;

    // don't add account, person, address as public property, because is set to null on construct !!

	/**
	 * The mongo collection where this document is stored
	 * @return string
	 */
	public function getCollectionName()
	{
		return 'mongocms_user';
	}

    /**
     * Returns the static model of the specified AR class.
     *
     * @return the static model class
     */
    public static function model($className = __CLASS__)
    {
        return parent::model($className);
    }

	/**
	 * Don't enable permissions for every user
	 *
	 * @return boolean
	 */
	public function exposePermissions()
	{
		return false;
	}

	/**
	 * Override from parent
	 * No page settings for user supported
	 *
	 * @return CForm
	 */
	public function createSettingsForm()
	{
	  return null;
	}

    /**
     * Add Account,Person,Address as embedded documents
     * The models will be rendered as tabs in the form
     *
     * @return array
     */
    public function embeddedDocuments()
    {
        return array(
            'account' => 'Account',
            'person' => 'Person',
            'address' => 'Address',
            );
    }


    /**
     *
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // don't forget to merge with parent rules
        return array_merge(parent::rules(), array(
                array('roles,account, person, address', 'safe',),
                array('roles,account, person, address', 'unsafe', 'on' => 'search, view',),
                ));
    }

    /**
     * All todo before saving to mongodb
     * beforeSave can't be used in embedded doc so we have to handle the account here
     */
     public function beforeSave()
    {
		if (!parent::beforeSave())
            return false;

        $this->account->loginname = strtolower($this->account->loginname);
     	$this->account->verifypassword = ''; //don't save the uncrypted pw

    	if (empty($this->account->password) && !empty($this->storedPW))
				$this->account->password = $this->storedPW;
     	else
		if (!empty($this->account->password))
    	   $this->account->password = $this->account->hashPassword($this->account->password);

     	//@see UserController.update
		if (isset($this->storedPW))
     		unset($this->storedPW);

     	if (empty($this->lastLogin))
     		$this->lastLogin = 0;

     	if (empty($this->lastOnline))
     		$this->lastOnline = 0;

     	if (empty($this->roles))
     		$this->roles = array(); //ensure to save as array in mongodb

        return true;
    }

	/**
	 * Return user roles as array (name => description)
	 * Only the name of the role is stored in user roles
	 * Checks if role really exists when assigning roles to Yii user on login
	 *
	 * @see MongoCmsUserIdentity.authenticate()
	 *
	 * @return array
	 */
	public function getRoles()
	{
		$roles = array();
		$definedRoles = Yii::app()->mongocmsDefinedRoles(false);
		foreach ($this->roles as $role)
			if (array_key_exists($role,$definedRoles))
			   $roles[$role] = $definedRoles[$role];

		return $roles;
	}

	/**
	 * Update last login timestamp in mongodb for this user
	 *
	 */
	public function updateLastLogin()
	{
		$criteria = array('_id' => new MongoId($this->_id));
		$values = array('lastOnline' => time());
		$this->atomicUpdate($criteria,$values);
	}

	/**
	 * Update last login timestamp in mongodb for this user
	 * @see MongoCmsBehavior.mongocmsUser
	 */
	public function updateLastOnline()
	{
		$criteria = array('_id' => new MongoId($this->_id));
		$values = array('lastOnline' => time());
		$this->atomicUpdate($criteria,$values);
	}

	/**
	 * Rename label for docroute from Route to Type
	 *
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array_merge(parent::attributeLabels(),
			         array(
					    	'docroute' => MongoCmsModule::t('Type'),
					    ));
	}


}
