<?php
/**
 * Page.php
 *
 * The base for all contenttypes
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class Page extends MongoCmsDocument
{
	const STATUS_INACTIVE = 0;
    const STATUS_ACTIVE = 1;

	const DOCROUTE_PREFIX_ATTR = 'docroute_prefix'; //softAttribute for handling docroute prefix
	const SETTINGSMETHOD_PREFIX = 'settings'; //the prefix for settings methods

	public $version; //the mongocms version
    public $modelclass;
    public $docroute;
	public $status;
    public $language;
    public $weight;
    public $title;
    public $subtitle;
    public $body;
    public $tags;
    public $validfrom;
    public $validto;
    public $attachments;
    public $created;
    public $modified;
    public $author;
    public $owner;
	public $ownerdocroute; //the users roles on saving new record
	public $ownerroles;    //the users docroute on saving new record
    public $permissions;
	public $externallinks;
	public $references;
	public $settings; //save the result of 'settingsNAME' methods
    public $data; //custom data

	private $_staticDocRoute;

    protected static $_authManager = array(); //cache for authManager

    /**
     * Returns the static model of the specified AR class.
     *
     * @return the static model class
     */
    public static function model($className = __CLASS__)
    {
        return parent::model($className);
    }


    /**
     * Assign configured mongodb from module
     * Assign modelclass, ... and SoftAtrributes
     *
     * @param string $scenario
     */
    public function __construct($scenario = 'insert')
    {
        parent::__construct($scenario);

        $connectionId = Yii::app()->controller->module->mongoConnectionId;
        $db = Yii::app()->getComponent($connectionId);
        $this->setMongoDBComponent($db);
        $this->references = array(); //important when using $addToSet in addReference

        $this->modelclass = $this->getModelClass(); //mayby overridden in other docs ?
        $this->attachEventHandler('onAfterSave', array($this, 'pageAfterSave'));
        $this->attachEventHandler('onAfterFind', array($this, 'pageAfterFind'));

        $extAttr = $this->getSoftAttributes();
        if (!empty($extAttr))
            $this->initSoftAttributes($extAttr);
    }

    /**
     * Initialize the default values
     *
     * @return
     */
    public function init()
    {
        $this->ensureIndexes = Yii::app()->mongocmsModule()->ensureIndexes;
        parent::init();

        if ($this->scenario == 'insert') {
            $this->validfrom = time();
            $this->validto = $this->getMaxValidTo();
            $this->status = self::STATUS_ACTIVE;
            $this->weight = 0;
        }
    }


	/**
	 * Define additional attributes in inherited classes
	 * DOCROUTE_PREFIX_ATTR for handling docroute attribute in forms / beforeSave
	 *
	 * @return array of string
	 */
	public function getSoftAttributes()
	{
		return array(self::DOCROUTE_PREFIX_ATTR);
	}

	/**
	 * Add EEmbeddedArraysBehavior
	 * @link http://canni.github.com/YiiMongoDbSuite/xhtml/basic.arrays.embedded-documents.html
	 *
	 * @return
	 */
	public function behaviors()
	{
		return array(
		    'embeddedArrays' => array(
		        'class'=>'ext.YiiMongoDbSuite.extra.EEmbeddedArraysBehavior',
		        'arrayPropertyName'=>'references',  // name of property, that will be used as an array
		        'arrayDocClassName'=>'Reference'    // class name of embedded documents in array
		    ),
		);
	}

	/**
	 * For displaying when choose pagetype
	 * Override in extented pagetypes
	 *
	 * @return string
	 */
	public function getModelClass()
	{
		return isset($this->modelclass) ? $this->modelclass : get_class($this);
	}

	/**
	 * Return the route to CRUDController
	 * Tries to find in controllerMap too
	 * Override in custom contenttypes if necessary
	 *
	 * @param boolean $addModulePath
	 *        if true: mongocms/ will be added as prefix
	 *        createUrl from module controllers will add this per default
	 *
	 * @param mixed $addModulePath
	 * @param string $subModuleId
	 * @param mixed $crudControllerId
	 * @return string
	 */
	public function getCRUDControllerRoute($crudControllerId = null,$addModulePath=true)
	{
		if ($this->isPage())
			 return Yii::app()->mongocmsControllerRoute('admin',$addModulePath);

		$modelClass = $this->getModelclass();

		if (empty($crudControllerId))
		    $crudControllerId = strtolower($modelClass);


		$moduleId = Yii::app()->mongocmsModuleFromContenttype($modelClass)->id;

		$subModuleId = $moduleId == 'mongocms' ? '' : substr(strrchr($moduleId, '/'),1);

		$result = Yii::app()->mongocmsControllerRoute($crudControllerId,$addModulePath,$subModuleId);

		//try to use controller from parent class
		if (empty($result))
		{
			$parentClass = get_parent_class($this);
			if ($parentClass !== false) {
			  $crudControllerId = strtolower($parentClass);
			  $result = Yii::app()->mongocmsControllerRoute($crudControllerId,$addModulePath,$subModuleId);
			}
		}

		if (empty($result))
		{
			$crudControllerId = 'admin'; //set to admincontroller
			$result = Yii::app()->mongocmsControllerRoute($crudControllerId,$addModulePath,$subModuleId);
		}

		return $result;
	}

    /**
     * Check if data of this model exists in the collection
     *
     * @return boolean
     */
    public function dbDataExists()
    {
        $criteria = array('modelclass' => $this->getModelClass());
        $cursor = $this->getCollection()->findOne($criteria, array('_id'));
        return !empty($cursor);
    }

	/**
	 * Check if the docroute of this modelclass exists in the collection
	 * Use this method when importing/createing data to avoid importing twice
	 *
	 * @see config/mongocms/default_data.php
	 *
	 * @return boolean
	 */
	public function docRouteExists($docroute,$modelClassOnly=true)
	{
		$criteria = array(
		                    'docroute' => $docroute,
	                      );

		if ($modelClassOnly)
			 $criteria['modelclass'] = $this->getModelClass();

		$cursor = $this->getCollection()->findOne($criteria, array('_id'));
		return !empty($cursor);
	}


    /**
     * What to in mongodb if installed
     *
     * @return
     */
    public function installContentType($recreate = false)
    {
        // save defaultpermissions for this contenttype to mongodb
        $this->installDefaultContenttypePermissions($recreate);
    }


	/**
	 * Create a MongoCmsAuthManager and add the permissions to be controlled
	 * Includes a install script, extracts var $authManager within the scripts
	 *
	 * @see config/mongocms/install_roleoperations_page.inc.php
	 *
	 * Searchpath of the installscript
	 * 1. install_roleoperations_MODELCLASS
	 * - modules config/THEME directory
	 * - if is submodule and not found: mongocms/config/THEME
	 * 2. install_roleoperations_MODELPARENTCLASS in these directories
	 * 3. install_roleoperations_page in these directories
	 *
	 * @return MongoCmsAuthManager
	 */
	public function createDefaultContenttypePermissions($authManager)
	{
		//delete from collection if exists
		$authManager->delete($this->getAuthId());

		$config = new MongoCmsConfiguration();
		$script = 'roleoperations_' . strtolower($this->getModelClass());
		$params = array('authManager' => $authManager); //$params

		if (!$config->includeScript($script,$params) && !$this->isPage())
		{
			$parentClass = get_parent_class($this);
			$script = 'roleoperations_'.strtolower($parentClass);
			if (!$config->includeScript($script,$params))
			{
				$script = 'roleoperations_page';
				$config->includeScript($script,$params);
			}
		}

		$authManager->save();
	}

    /**
     * Check if permissions for this contenttype are installed
     *
     * @return
     */
    protected function installDefaultContenttypePermissions($recreate = false)
    {
        $authManager = $this->getContenttypeAuthManager(!$recreate);
        if ($recreate || !$authManager->contentTypeExists())
            $this->createDefaultContenttypePermissions($authManager);
    }

    /**
     * What to do in mongodb if unInstalled
     *
     * @return
     */
    public function uninstallContentType()
    {
        // delete contenttypepermissions from mongodb
        $this->getContenttypeAuthManager()->delete();
    }

    /**
     * What to do in mongodb if code changed
     *
     * @return
     */
    public function updateContentType()
    {
        //  @todo
    }

	/**
	 * Set the static route by $_GET param
	 * @see AdminController.create
	 * @see getStaticDocRoutes
	 *
	 * @param string $route
	 */
	public function setStaticDocRoute($route){
		$this->_staticDocRoute = $route;
	}

	/**
	 * Get the static (fixed) docroutes
	 * If not empty, the user can select one in a drowdown list
	 * on update/insert page
	 *
	 * These static routes can be configured in config/mongocms/docroutes_modelclass.php
	 *
	 * @see MongoCmsUtil::docrouteFormElements
	 * @see config/mongocms/docroutes_slideshow
	 * @return array
	 */
    public function getStaticDocRoutes()
    {
    	if (isset($this->_staticDocRoute))
    		return array($this->_staticDocRoute => $this->_staticDocRoute);

    	$modelClass = $this->getModelClass();

		$config = new MongoCmsConfiguration();
    	$name = 'docroutes_' . strtolower($modelClass);

    	$module = Yii::app()->mongocmsModuleFromContenttype($modelClass);

    	if (empty($module))
    		throw new CException('Implementation error of mongocmsModuleFromContenttype');

    	if ($config->loadModuleConfig($module, $name))
    		return $config->toArray();
    	else
    		return null;
    }


	/**
	 * Create the form for the page settings
	 * This form is shown in a tab of the form view
	 * The corresponding model is 'PageSettingsForm'
	 * Override for your needs in other contenttypes:
	 * return null if no settings provided
	 *
	 * @see PageSettings.php
	 *
	 * @return CForm
	 */
	public function createSettingsForm()
	{
		$model = new PageSettings();
		if (!$this->getIsNewRecord())
			$model->setAttributes($this->settings);

		$config = array(

				    //'ActiveForm' without output of formbegin and formend
					'activeForm'=>array('class'=>'MongoCmsEmbeddedForm'),

					//define the input elements for the public properties
					'elements' => array(

                        'metaKeywords' => array(
				            'type' => 'text',
				            'size' => 60,
				            'hint' => MongoCmsModule::t("Keywords for HTML metatag 'keywords'"),
				            ),

						'metaDescription' => array(
							'type' => 'text',
							'size' => 60,
							'hint' => MongoCmsModule::t("Text for HTML metatag 'description'"),
						),

						'showAuthor' => array(
							'type' => 'checkbox',
							'hint' => MongoCmsModule::t("Enable to show the author in the page view"),
						),

						'showCreated' => array(
							'type' => 'checkbox',
							'hint' => MongoCmsModule::t("Enable to show the created timestamp in then page view"),
						),

						'showModified' => array(
							'type' => 'checkbox',
							'hint' => MongoCmsModule::t("Enable to show the modified timestamp in page view"),
						),

						'showPrintPage' => array(
							'type' => 'checkbox',
							'hint' => MongoCmsModule::t("Enable to show the 'Print page' link"),
						),

                       'showFavoritesLink' => array(
						    'type' => 'checkbox',
						    'hint' => MongoCmsModule::t("Enable to show the 'Add to favorites' link"),
						    ),

                       /* not implemented yet

						'showFollowLink' => array(
						    'type' => 'checkbox',
						    'hint' => MongoCmsModule::t("Enable to show the 'Follow' link"),
						    ),

						'allowComments' => array(
						    'type' => 'checkbox',
						    ),
		                 */
				        ),
				    );

		return new CForm($config, $model);
	}

	/**
	 * Checks if a page settings item exists
	 *
	 * @param string $item
	 * @return boolean
	 */
	public function hasSetting($item)
	{
		if (empty($this->settings) || empty($item))
			return false;
		else
		    return array_key_exists($item,$this->settings);
	}


	/**
	 * Return the value of a specific page setting
	 *
	 * @param string $item
	 * @return mixed
	 */
	public function getSettingsValue($item)
	{
		return $this->hasSetting($item) ? $this->settings[$item] : null;
	}

	/**
	 * Allow set content permissions for this contenttype
	 *
	 * @return boolean
	 */
	public function exposePermissions()
	{
		return true;
	}

	/**
	 * Check if reference exists
	 *
	 * @param mixed $id
	 * @return boolean
	 */
	public function hasReference($id)
	{
		$exists = false;
		if (is_array($this->references))
			foreach ($this->references as $reference)
				if ($reference['id'] == $id)
				{
					$exists = true;
					break;
				}
		return $exists;
	}

	/**
	 * Register a reference to another model
	 *
	 * @param string $title
	 * @param string $type
	 * @param string $id
	 * @param string $modelclass
	 */
	public function addReference($title,$type,$id,$modelclass,$modified)
	{
		$reference = new Reference();

		$reference->title = (string)$title;
		$reference->type = (string)$type;
		$reference->id = (string)$id;
		$reference->modelclass = (string)$modelclass;
		$reference->modified = (int)$modified;
		$reference->added = time();

	    //append in memory
		if (!$this->hasReference($id))
		   $this->references[] = $reference;

		//atomic update: add new references if not exists
		$action = array('$addToSet' => array('references' => $reference->toArray()));
		$criteria = array('_id' => new MongoID($this->_id));
		$this->getCollection()->update($criteria, $action);

	}

	/**
	 * Remove a reference from a model
	 *
	 * or this reference from all models
	 * @see beforeDelete
	 *
	 *
	 * @param string $id
	 * @param boolean $removeAll
	 */
	public function removeReference($id,$fromAllModels=false)
	{
		//remove from memory
		$removeIdx = -1;
		if ($this->hasReference($id))
		 for($idx=0; $idx < count($this->references); $idx++)
			if ($this->references[$idx]['id'] == $id)
			{
				$removeIdx = $idx;
				break;
			}

		if ($removeIdx >= 0)
			unset($this->references[$removeIdx]);

		//atomic update: remove reference array with id = $id

		//IMPORTANT convert explizit to string: (string)$id
		$action = array('$pull' => array('references' => array('id'=>(string)$id)));

		$criteria = $fromAllModels ? array('references.id' => (string)$id)
			                       : array('_id' => new MongoID($this->_id));

		//need multiflag to remove all, not the first found only on $removeAll!
	    $options = $removeAll ? array('multiple'=>true) : array();
		$this->getCollection()->update($criteria,$action,$options);
	}

	/**
	 * Return the visible attributes in an action context
	 * This is used for lists (grid, index ...) but not in a or a single view
	 *
	 * @param mixed $scenario
	 * @return array
	 */
	public function getVisibleAttributes($actionId = null)
	{
	  if (!isset($action))
	  		$actionId = Yii::app()->controller->action->id;

	  switch($actionId){
	  		case 'admin':
	  			return array('docroute','title','status','author');
	  			break;

	  		default:
	  			return array('docroute','status','language','tags',
	  			             'author','created','modified',
	  			             'validfrom','validto',
	  			             'title','subtitle','body');
	  	}
	}

    /**
     * These hints will be displayed in the edit form
     *
     * @return
     */
    protected function getAttributeHints()
    {
        return array(
                'docroute' => MongoCmsModule::t('Set the route of this node'),
                'tags' => MongoCmsModule::t('Separate multiple tags with comma'),
            );
    }

    /**
     * Display a hint for an attribute in the form
     * The hints can be defined in getAttributeHints()
     *
     * @param string $attribute
     * @param string $tag
     * @param array $htmlOptions
     * @return
     */
    public function getAttributeHint($attribute, $tag = 'span', $htmlOptions = array())
    {
        static $hints;

        if (!isset($hints))
            $hints = $this->getAttributeHints();

        if (array_key_exists($attribute, $hints))
            return CHtml::tag($tag, $htmlOptions, $hints[$attribute]);

        return '';
    }

    /**
     * Returns a teaser created from body with strlen <= mongocms module teaserSize
     *
     * @param boolean $truncated returns true if $truncated
     * @return string
     */
    public function getTeaser($readMoreLabel = null,$readMoreOnNotTrunctated = false)
    {
        $truncated = false;
        $separatorChars = " \n>";
        $teaser = substr($this->body, 0, Yii::app()->mongocmsModule()->teaserSize);

        if (!empty($teaser))
        {
            $reversedTeaser = strrev($teaser);
            if (($truncStr = strpbrk($reversedTeaser, $separatorChars)) !== false)
            {
				$teaser = substr($teaser, 0, strlen($truncStr));
                $truncated = true;
            }
        	if ($truncated)
        		$teaser .= '... ';

			if (!empty($readMoreLabel) && ($readMoreOnNotTrunctated || $truncated))
               $teaser .= '<br/>' . $this->getPageLink($readMoreLabel);
        }

        return $teaser;
    }

    /**
     * Used for contenttype permissions
     * This is the authId for MongoCmsAuthManager
     * Every contenttype has an own MongoCmsAuthManager in mongodb
     *
     * @return string
     */
    public function getAuthId()
    {
        return $this->getModelClass();
    }

    /**
     * Get the AuthManager of this page
     * Can be for the class only (contenttype permission)
     * or for an instance (per instance permission)
     *
     * @param boolean $asContentType
     * @return MongoCmsAuthManager
     */
    public function getContenttypeAuthManager($loadFromDB = true)
    {
        $authId = $this->getAuthId();

       	if (isset(self::$_authManager[$authId][(integer)$loadFromDB]))
       		return self::$_authManager[$authId][(integer)$loadFromDB];

        $authManager = new MongoCmsAuthManager($authId);
        if ($loadFromDB)
            $authManager->init();

        Yii::trace('Authmanager created: "' . $authManager->authId .
            ' loadFromDB: ' . (integer)$loadFromDB,
            'mongocms.models.Page.getContenttypeAuthManager');

        return self::$_authManager[$authId][(integer)$loadFromDB] = $authManager;
    }

    /**
     * Save the permissions submitted from permissions form
     *
     * @see AdminController: actionPermissions,actionUpdate, actionCreate
     * @see views/admin/_form_permissions, views/admin/_form_permissions_item,
     * @param array $formdata
     */
    public function saveContenttypePermissions($formdata)
    {
		//load from db
		$authManager = $this->getContenttypeAuthManager();
        // reset
        $authManager->removeRoleOperations();
        // add permissions: structure @see views/admin/_permissions.php
        foreach ($formdata as $operation => $roles)
	        foreach ($roles as $role)
	        {
	            $item = $authManager->getAuthItem($operation);
	            if (empty($item))
	                $authManager->createOperation($operation);

	            $item = $authManager->getAuthItem($role);
	            if (empty($item))
	                $authManager->createRole($role);

	            $authManager->addItemChild($role, $operation);
	        }

        $authManager->save();
    }

    /**
     * Denies the access of the user.
     * This method is invoked when access check fails.
     */
    public static function accessDenied()
    {
        $user = Yii::app()->user;
        if ($user->getIsGuest())
            $user->loginRequired();
        else
			throw new CHttpException(403, MongoCmsModule::t('Access denied.'));
    }


    /**
     * Check access to this contenttype, configured in the corresponding authmanager
     *
     * @see MongoCmsUserIdentity->authenticate()
     * @param mixed $itemNames : string or array; if array all items must match
     * @param array $params the params for calculating the bizRule
     * @param boolean $checkOnly : no accessDenied, if not valid access; used for building menu
     * @param boolean $allowCaching
     * @param boolean $or true: access is allowed if at least one operation of the array $itemNames is allowed,
     *                    false: all operations have to be allowed
     * @return boolean
     */
    public function checkContenttypeAccess($itemNames, $params = array(), $checkOnly = false, $or = true)
    {
		$authManager = $this->getContenttypeAuthManager();
        return $authManager->checkUserAccess($itemNames, $params, $checkOnly, $or);
    }

    /**
     * Check if permission for a role/operation is set
     *
     * @see views/_form_permissions_content
     * @param mixed $role
     * @param mixed $operation
     * @return boolean
     */
    public function hasPermission($role, $operation, &$hasContentPermissions)
    {
        if (empty($role) || empty($operation))
            return false;

        $hasContentPermissions = false;
        if (empty($this->permissions) || empty($this->permissions['roles']))
            return true;

        $hasContentPermissions = true;
        return !empty($this->permissions['roles'][$operation]) &&
        in_array($role, $this->permissions['roles'][$operation]);
    }

    /**
     * Check access to this loaded instance from mongodb
     * If itemNames is an array, all items have to be allowed
     *
     * @param mixed $itemNames string or array of operations
     * @param boolean $checkOnly abort on accessDenied or not
     * @param boolean $or true: access is allowed if at least one operation of the array $itemNames is allowed,
     *                    false: all operations have to be allowed
     * @return boolean
     */
    public function checkContentAccess($itemNames, $checkOnly = false, $or = true)
    {
        if (!$this->exposePermissions())
            return true;

        if (empty($itemNames))
            return $checkOnly ? false : $this->accessDenied();

        // if no permissions assign to this content, all operations are allowed
        if (empty($this->permissions) || empty($this->permissions['roles']))
            return true;

        // all access allowed for root, even if invalid item
        //  @see MongoCmsUserIdentity->authenticate(), config/default_users.php
        if (Yii::app()->mongocmsIsRootUser())
            return true;

        $userRoles = Yii::app()->mongocmsCurrentUserRoles(); //array ($role=>$description, ...)

        if (is_string($itemNames))
        {
            foreach ($userRoles as $role => $description)
            {
                if ($this->hasPermission($role, $itemNames, $permissionsSet))
                    return true;
            }

            return $checkOnly ? false : $this->accessDenied();
        }
        else
		{
        	for($i = 0; $i < count($itemNames); $i++)
        	{
        		$allowed = false;
				foreach ($userRoles as $role => $description)
        		{
        		   if ($this->hasPermission($role, $itemNames[$i], $permissionsSet))
        				$allowed = true;

        			Yii::trace('Checked content access "' . $itemNames[$i] .
        				'" role: ' .$role .
        				' allowed: ' .$allowed);

        			if ($allowed && $or)
        				return true;
        		}

        		if (!$or && !$allowed) //AND: all have to be allowed
        			return $checkOnly ? false : Page::accessDenied();
        	}

			return $or ? ($checkOnly ? false : Page::accessDenied()) : true;
        }
    }

    /**
     * 1. Check 'access' to docroute of the model
     * 2. Check ContentAccess for operations $itemNames
     * 3. Check ContenttypeAccess for operations $itemNames
     *
     * @param mixed $itemNames string or array of operations
     * @param boolean $params the params for calculating the bizRule
     * @param boolean $checkOnly abort on accessDenied or not
     * @param boolean $or true: access is allowed if at least one operation of the array $itemNames is allowed,
     *                    false: all operations have to be allowed
     * @return boolean
     */
    public function checkAccess($itemNames, $params = array(), $checkOnly = false, $or = true)
    {
        if (isset($this->docroute)) //only check if model is fully loaded
        {
        	if (!DocRoute::model()->checkDocRouteAccess($this->docroute,$checkOnly))
        		return false;
        }

		if (!$this->checkContentAccess($itemNames, true, $or))
            return $checkOnly ? false : $this->accessDenied();

        if (!$this->checkContenttypeAccess($itemNames, $params, true, $or))
            return $checkOnly ? false : $this->accessDenied();

        return true;
    }

    /**
     * Returns the available operations for checking contentaccess
     * These operations can be added to permissions
     * Override for your needs in other contenttypes (@see DocRoute.php)
     *
     * @return array
     */
    public function getContentAccessOperations()
    {
        return array('view' => MongoCmsModule::t('View'),
                     'update' => MongoCmsModule::t('Edit'),
                     'delete' => MongoCmsModule::t('Delete'),
                    );
    }

    /**
     * Returns the view to be rendered by controller
     * For submodules (controller extends AdminController):
     *
     * Search order - Look for:
     * 1. the view in controllers viewPath with appended classname (view_classname)
     * 2. the view in controllers viewPath
     * 3. the same in AdminControllers viewPath
     *
     * So no need for a submodule to implement
     * the views: create, admin, index, view ...
     *
     * @param mixed $view
     * @param string $suffix
     * @return string
     */
    private function _getView($view, $suffix = '', $controller = null)
    {
        $suffix = strtolower($suffix);
        if (!isset($controller))
            $controller = Yii::app()->controller;

        if (!empty($suffix))
            $view = $view . '_' . $suffix;

    	// look for view_classname in controllers viewPath
    	$classview = $view . '_' . strtolower($this->getModelClass());
    	if ($viewfile = $controller->getViewFile($classview))
    		return $classview;

		// look for the viewfile in controllers viewPath
        if ($viewfile = $controller->getViewFile($view))
            return $view;

        // if not exists and controller is not the AdminController
        // look for the viewfile in mongocms AdminControllers viewPath
        if (get_class($controller) != 'AdminController') {
            $route = 'mongocms/admin';
            $controllerArray = Yii::app()->createController($route);
            $adminController = $controllerArray[0]; //why an array?

        	if ($viewfile = $adminController->getViewFile($classview)) {
        		$view = '//../modules/mongocms/themes/' .
        		Yii::app()->theme->name . '/views/mongocms/admin/' . $classview;
        		return $view;
        	}

            if ($viewfile = $adminController->getViewFile($view)) {
                $view = '//../modules/mongocms/themes/' .
                Yii::app()->theme->name . '/views/mongocms/admin/' . $view;
                return $view;
            }
        }
        return $view;
    }

    /**
     * Returns the form to be rendered by controller
     * Search in lowercase(classname) subdirectory  (user/_form.php)
     * Search for form of embedded documents (user/_form_person.php)
     */
    public function getViewForm($suffix = '', $controller = null)
    {
        return $this->_getView('_form', $suffix, $controller);
    }

    /**
     * Returns the view to be rendered by controller
     * Returns the matching view of AdminController
     * if the view not exists in own viewPath
     */
    public function getView($scenario = null, $suffix = null, $controller = null)
    {
        if (!isset($scenario))
            $scenario = $this->scenario;

        return $this->_getView($scenario, $suffix, $controller);
    }

	/**
	 * Atomic update of the record
	 * @see documentation from mongodb
	 * note: per default only the first found record will be updated
	 *       if not multiflag is set
	 *
	 * @param array $criteria
	 * @param array $values
	 * @param string $modifier
	 * @param boolean $multiple
	 * @return
	 */
	public function atomicUpdate($criteria,$values,$modifier = '$set', $multiple = false)
	{
		if (empty($values))
			return false;

		$action = array($modifier => $values);

		$options = $multiple ? array('multiple'=>true) : array();
		return $this->getCollection()->update($criteria,$action,$options);
	}

    /**
     * Direct db command call
     *
     * @param array $command
     * @return
     */
    public static function dbCommand($command)
    {
        return self::model()->getDb()->command($command);
    }
    /**
     * Distinct query for a specific attribute in the collection
     *
     * @param string $attribute
     * @return array
     */
    public function getDistinct($attribute)
    {
        $command = array("distinct" => $this->getCollectionName(), "key" => $attribute);
        $result = self::dbCommand($command);
        return $result['ok'] = 1 ? $result['values'] : null;
    }

    /**
     * Predefined scopes for the ContentController/DownloadController
     *
     * @return
     */
    public function scopes()
    {
        return array(
            'active' => array(
                'conditions' => array(
                    'status' => array('==' => 1),
                    ),
                'sort' => array(
                    'weight' => EMongoCriteria::SORT_ASC,
                    'docroute' => EMongoCriteria::SORT_ASC,
                    'title' => EMongoCriteria::SORT_ASC,
                    ),
            ),

            'validFromTo' => array(
                'conditions' => array(
                    'validfrom' => array('<=' => time()),
                    'validto' => array('>=' => time()),
                    ),
            ),
        );
    }

    /**
     * named scope: byContentType
     *
     * @link http://canni.github.com/YiiMongoDbSuite/xhtml/advanced.named-scopes.html
     * @param string $modelClass
     * @return
     */
    public function byContentType($modelClass = null)
    {
        if (!isset($modelClass))
            $modelClass = $this->getModelClass();

        $criteria = $this->getDbCriteria();
        $criteria->modelclass = $modelClass;
    	$criteria->sort('docroute', EMongoCriteria::SORT_ASC);
        $this->setDbCriteria($criteria);

        return $this;
    }

    /**
     * Create an instance of a specific page
     *
     * @param string $modelclass : the class to create
     * @param array $attributes : the attributes to assign
     * @param string $scenario
     * @return EMongoSoftDocument
     */
    public static function createInstance($modelClass, $attributes = null, $scenario = null)
    {
        if (empty($modelClass))
            throw new CHttpException(400, 'Invalid request.');

        $model = new $modelClass($scenario);
        if (!($model instanceof Page))
            throw new CHttpException(400, 'Invalid request.');

        if (isset($attributes))
            $model->setAttributes($attributes, false);

        if (!empty($model->softAttributes))
        {
            $softAttributes = $model->getSoftAttributeNames();

            if (!empty($softAttributes))
                foreach ($softAttributes as $attribute)
                $model->$attribute = $attributes[$attribute];
        }

        return $model;
    }

    /**
     * Load an instance from mongodb
     *
     * @param mixed $id
     * @return
     */
    public function loadInstance($id)
    {
        return $this->findByPk(new MongoID($id));
    }

    /**
     * Helper function for use in views,
     * when using different contenttypes/modelclasses
     *
     * @param mixed $modelclass
     * @return
     */
    public function isModelClass($modelclass)
    {
        return $modelclass == $this->getModelClass();
    }

    /**
     * Helper function for use in views
     *
     * @see views/admin/create.php
     * @return
     */
    public function isPage()
    {
        return $this->getModelClass() == 'Page';
    }

    /**
     * Returns the possible status items
     *
     * @return array
     */
    public static function getStatusItems()
    {
        return array(
            Page::STATUS_INACTIVE => MongoCmsModule::t('Inactive'),
            Page::STATUS_ACTIVE => MongoCmsModule::t('Active'),
            );
    }

    /**
     * Returns a datestring, format configured in module
     *
     * @param unix $ timestamp $timestamp
     * @return string
     */
    public function getFormatedDate($timestamp)
    {
        return date(Yii::app()->controller->module->getDateFormat(), $timestamp);
    }

    /**
     * Returns a datetimestring, format configured in module
     *
     * @param unix $ timestamp $timestamp
     * @return string
     */
    public function getFormatedDateTime($timestamp)
    {
        return date(Yii::app()->controller->module->getDateTimeFormat(), $timestamp);
    }

    /**
     * 20 years in the future if empty, should be enough ;-)
     * can't add 100 years :-(
     *
     * @return unix timestamp
     */
    protected function getMaxValidTo()
    {
        return (integer)(time() + (20 * 365 * 24 * 60 * 60));
    }

	/**
	 * Helper function to remove slashes at the beginning and the end
	 *
	 * @param mixed $route
	 * @return
	 */
	public static function trimSlashes($route)
	{
		return trim($route,' /');
	}

    /**
     * All todo before saving to mongodb
     */
    public function beforeSave()
    {
        if (!parent::beforeSave())
            return false;

    	$this->version = Yii::app()->mongocmsVersion();

        $this->modelclass = $this->getModelClass();

        // set modified to now
        $this->modified = time();
        // need to convert, because of dropdown??
        $this->status = (integer)$this->status;

        // datetimes are stored as unix timestamp (for searching)
    	if(empty($this->validfrom))
    		$this->validfrom = 0;

    	if (is_string($this->validfrom))
    		$this->validfrom = strtotime($this->validfrom);

    	if(empty($this->validto))
    		$this->validto = $this->getMaxValidTo();

    	if (is_string($this->validto))
    		$this->validto = strtotime($this->validto);


        if (empty($this->weight))
            $this->weight = 0;
        else
            // need to convert, because of dropdown??
            $this->weight = (integer)$this->weight;


    	//tags to array
    	if (empty($this->tags))
    		//because of searching: @see MongoCmsContentProvider.getTags
    		$this->tags = null;
    	else
    	{
    		$tags = $this->tagsToArray();
    		$this->setAttributes(array('tags' => $tags),false);
    	}

        if (!$this->exposePermissions())
            $this->permissions = null;

    	//handling of docroute prefix
    	//softattribute submitted from create/update form
    	//add the selected prefix
    	//@see DocRouteController.actionFormElements, MongoCmsUtil::docrouteFormElements

    	if (empty($this->docroute))
    		$this->docroute = $this->getStaticDocRoutes();

    	$this->docroute = !empty($this->docroute) ? self::trimSlashes($this->docroute) : null;

    	$docRoutePrefix = self::DOCROUTE_PREFIX_ATTR;

    	if (!empty($this->$docRoutePrefix))
    	{
    		$value = self::trimSlashes($this->$docRoutePrefix);
    		if (strpos($this->docroute,$value) !== 0)
    			$this->docroute = empty($this->docroute) ? $value : $value . '/' . $this->docroute;
    	}

    	//don't save this softattribute
		if (isset($this->$docRoutePrefix))
    		unset($this->$docRoutePrefix);

        //set created, owner,author if is new record
        if ($this->isNewRecord)
        {
            $author = empty(Yii::app()->user->name) ? 'guest' : Yii::app()->user->name;
            $this->setAttributes(
                array(
                    'created' => time(),
                    'author' => $author,
                    'owner' => Yii::app()->user->id,
                    ),
                false // no need for validation
                );

        	//save the docroute and roles of the current user
        	//usage in permissions for updateSameUserType and updateSameRole
        	//@see config/mongocms/roleoperations_page.php
        	$userModel = Yii::app()->mongocmsUser();
        	if (!empty($userModel)) {
        		$this->setAttributes(
        		array(
        		    'ownerdocroute' => $userModel->docroute,
        		    'ownerroles' => $userModel->roles,
        		    ),
        		false
        		);
        	}
        	else
        	{
        		$this->setAttributes(
        		array(
        		    'ownerdocroute' => '',
        		    'ownerroles' => array(),
        		    ),
        		false
        		);
        	}
        }

        $this->rebuildAttachments();
        // important: do after rebuildAttachments
        $this->saveNewAttachments();

        return true;
    }

	/**
	 * Get the allowed file extensions for attachments
	 * from the global config of mongocms module
	 * Converst array to 'accept' property  string of MultiFileUpload per default
	 *
	 * @param boolean $asAcceptFileUpload
	 * @return mixed
	 */
	public function getAllowedFileExtensions($asAcceptFileUpload=true)
	{
		$extArray = Yii::app()->mongocmsModule()->allowedFileExtensions;

		return $asAcceptFileUpload ? implode('|',$extArray) : $extArray;
	}

	/**
	 * Convert comma separated tags into array
	 *
	 * @return array
	 */
	public function tagsToArray()
	{
		$tags = array();
        if (!empty($this->tags))
        {
        	$tagItems = MongoCmsUtil::splitWords($this->tags);

        	if (!empty($tagItems))
	        	foreach ($tagItems as $tagItem)
	        		$tags[]=trim($tagItem);
        }

		return $tags;
	}

	/**
	 * Convert array tags into comma separated string
	 * @see pageAfterFind
	 *
	 * @return array
	 */
	public function tagsToString()
	{
		if (is_array($this->tags) && count($this->tags))
			return implode(',',$this->tags);
		else
			return null;
	}


    /**
     * Delete the attached files of the page from GridFs
     * Delete (maybe) docroute entry from DocRoute
     */
    protected function beforeDelete()
    {
        if (parent::beforeDelete()) {
            // delete attached files
            foreach ($this->attachments as $attachment) {
                $fileid = $attachment['_id'];
                $file = $this->attachedFile($fileid);
                if (isset($file))
                    $file->delete();
            }

        	//remove all references to this from other page
        	$this->removeReference($this->_id,true);

        	//remove all references to this from user
        	User::model()->removeReference($this->_id,true);

            // delete record DocRoute if the deleted item is the only one with this path
        	// and the attribute docrouteid is not set in DocRoute
            $criteria = array('docroute' => $this->docroute,
                              'docrouteid' => null);

            if ($this->getCollection()->count($criteria) == 1)
            {
                $criteria = array('docroute' => $this->docroute,
                                  'itemCollectionName' => $this->getCollectionName(),
                    );
                $collection = DocRoute::model()->getCollection();
                $collection->remove($criteria);
            }
            return true;
        } else
            return false;
    }

    /**
     * Don't register all attachment fields in $this->attachments
     *
     * @param MongoCmsFile $file
     * @return array
     */
    protected function getAttachmentDescriptionFields($file)
    {
        $regFields = $file->toArray();
        unset($regFields['permissions']); //don't register twice
        unset($regFields['type']); //is always "attachment"
        unset($regFields['filename']); //no need for temppath to file
        return $regFields;
    }

    /**
     * Saves the uploaded attachments to GridFS (@see: components/MongoCmsFile.php)
     * Generates images from preset before save (@see: MongoCmsFile->3())
     * Saves the _id and attachment description (_id, orgfilename, metadata, imagepresets) in the the array attachments
     */
    protected function saveNewAttachments()
    {
        $newAttachments = array();

        $modelClass = $this->getModelClass();
        $module = Yii::app()->mongocmsModule();

        if ($attachmentfiles = CUploadedFile::getInstancesByName('new_attachments'))
            foreach ($attachmentfiles as $file)
            {
	            $attachment = new MongoCmsFile();

	            $attachment->loadFromFile($file->tempName, $file->name);
	            // copy own permissions to the attachment
	            $attachment->setAttributes(array(
	                    'ownerpermissions' => $this->permissions,
	                    'ownermodelclass' => $modelClass,
	                    ),
	                false // are marked unsafe
	                );
	            // assign the config of the image presets to the attachment
	            // Searchpath for config see MongoCmsConfiguration
	            // the images from preset will be generated on save
	            if ($attachment->isImage())
	            {
	                $config = new MongoCmsConfiguration();
	            	$module = Yii::app()->mongocmsModuleFromContenttype($modelClass);

	                if ($config->loadModuleConfig($module, 'imagepresets') &&
	                        isset($config[$modelClass]))
	                    $attachment->setImagePresetsConfig($config[$modelClass]);
	            }

	            if ($attachment->save())
	            {
	                $newAttachments[] = $this->getAttachmentDescriptionFields($attachment);
	                // remove uploaded file
	                unlink($attachment->filename);
	            }
	        }
        // copy the registered fields to $this->attachments
        $existingAttachments = is_array($this->attachments) ? $this->attachments : array();
        $this->setAttributes(array('attachments' => array_merge($newAttachments, $existingAttachments)), false);
    }

    /**
     * Rebuilds the attachments by the submitted _id from the update form
     * Because of possible inconsitent attributes the update form is only populated
     * with the related _id from the GridFS
     * When the flag "remove" (from the checkbox) is set, the attachment will be
     * deleted in GridFS and not be rebuilt
     *
     * @see MongoCmsFileWidget
     * @return
     */
    protected function rebuildAttachments()
    {
        if ($this->isNewRecord)
            return;

        $rebuiltAttachments = array();

        foreach ($this->attachments as $attachment) {
            $fileid = $attachment['_id'];
            $file = $this->attachedFile($fileid);

            if (empty($file)) // unknown error: $file is not instance of MongoCmsFile
                continue;

            if ($attachment['remove'])
                $file->delete();
            else {
                // update the file->permissions if this->permissions have changed
                $file->updatePermissions($this->permissions);
                $rebuiltAttachments[] = $this->getAttachmentDescriptionFields($file);
            }
        }

        $this->unsetAttributes(array('attachments'));
        $this->setAttributes(array('attachments' => $rebuiltAttachments), false);
    }

    /**
     * Get the related MongoCmsFile from GridFS
     * if imagePreset is set, return the image preset file
     *
     * @param string $fileid
     * @return MongoCmsFile
     */
    public function attachedFile($fileid, $imagePreset = null)
    {
        // search for the named imagePreset, set fileid to the preset if exists
        if (!empty($imagePreset)) {
            foreach ($this->attachments as $attachment) {
                if ($attachment->_id == $fileid &&
                    MongoCmsUtil::isImage($attachment->metadata['type']) &&
                        isset($attachment['imagepresets']) &&
                        isset($attachment['imagepresets'][$imagePreset])) {
                    $fileid = $attachment['imagepresets'][$imagePreset];
                    break;
                }
            }
        }

        $file = MongoCmsFile::model()->findByPk(new MongoID($fileid));

        if ($file instanceof MongoCmsFile)
            return $file;
    }

    /**
     * Get the viewUrl of an attachment if is an image
     * If imagepreset not exists, the url to the original image will be returned
     *
     * @see MongoCmsFileWidget
     * @param array $attachment
     * @param string $imagePreset
     * @param boolean $presetOnly , if false, return original image if preset not exists
     * @return string or false
     */
    public function getAttachmentImageUrl($attachment, $imagePreset, $presetOnly = false)
    {
        if (MongoCmsFile::isImageMimeType($attachment['metadata']['type'])) {
            $imgFileId = null;
            // check for thumbnail
            if (isset($attachment['imagepresets']) &&
                    isset($attachment['imagepresets'][$imagePreset]))
                $imgFileId = $attachment['imagepresets'][$imagePreset];
            else
            if (!$presetOnly)
                $imgFileId = $attachment['_id']; //orgfile
            if (!empty($imgFileId))
                return MongoCmsFile::viewUrl($imgFileId);
        }

        return false;
    }

    /**
     * Returns an array of url of a specific preset
     *
     * @param string $presetName
     * @return array
     */
    public function getAttachmentImageUrls($presetName)
    {
        $result = array();
        if (is_array($this->attachments))
            foreach ($this->attachments as $attachment) {
            $presetUrl = $this->getAttachmentImageUrl($attachment, $presetName, true);
            if ($presetUrl !== false)
                $result[] = $presetUrl;
        }
        return $result;
    }

	/**
	 * Returns the url path for the action: create, delete ...
	 * Contenttype access for the action is checked
	 *
	 * @param string $action
	 * @param array $params
	 * @return url
	 */
	public function getActionItemUrl($action,$controllerId = null,$subModuleId='',$params = array())
	{
		if (!$this->checkContenttypeAccess($action,array(),true))
			return false;

	    $route = $this->getCRUDControllerRoute($controllerId,true,$subModuleId);

		if (!empty($route))
		{
			//add modelclass if AdminController is used an not is page
			if ($route == 'admin' && !$this->isPage())
			      $params = array_merge($params,array('modelClass'=>$this->getModelClass()));

			return Yii::app()->createUrl($route .'/'.$action,$params);
		}

		return false;
	}

	/**
	 * Returns the url for a full public view of the page
	 * This is the url of ContentController.actionPage
	 *
	 * @param array $params
	 * @param boolen $checkOnly
	 * @return string
	 */
	public function getPageUrl($checkOnly=true,$id = null)
	{
		if (!$this->checkAccess('view',array(),$checkOnly))
			return '';

		$route = Yii::app()->mongocmsControllerRoute('content') .'/page';
		//need to add id as extra param because of ContentController.actionPage
		$pageId = isset($id) ? $id : $this->_id;
        return Yii::app()->createUrl($route .'/'. $this->docroute) .'?id='.$pageId . '&modelClass='.$this->getModelClass();
	}

	/**
	 * Returns a link to the full view of the page
	 * This is the link to the ContentController.actionPage
	 *
	 * @param array $params
	 * @param boolen $checkOnly
	 * @return string
	 */
	public function getPageLink($label,$id = null,$htmlOptions = array())
	{
		$url = $this->getPageUrl(true,$id);
		return empty($url) ? '' : CHtml::link(MongoCmsModule::t($label),$url,$htmlOptions);
	}

    /**
     * Register docroute in mongodb
     * Set the owner of the attachments to $this->_id
     *
     * @param mixed $event
     */
    public function pageAfterSave($event)
    {
        $this->registerDocRoute($event);

    	//flush menu cache @see MongoCmsBehavior
    	Yii::app()->mongocmsMenuCache->flush();
    	//flush page cache @see MongoCmsBehavior
    	Yii::app()->mongocmsPageCache->flush();
    }


	/**
	 * Convert tags array to string
	 *
	 * @param mixed $event
	 */
	public function pageAfterFind($event)
	{
		$this->tags = $this->tagsToString();
	}

    /**
     * Save the current docroute into the docroute collection
     *
     * @return boolean ;
     */
    protected function registerDocRoute($event)
    {
        if (!($event->sender instanceof Page))
            return false;

        if (empty($this->docroute))
            return false;

    	$collectionName = $this->getCollectionName();

    	$docRouteModel = DocRoute::model();
        // don't save docroute itself
        if ($collectionName == $docRouteModel->getCollectionName())
            return false;

        // avoid recursion
        $this->detachEventHandler('onAfterSave', array($this, 'registerDocRoute'));

        $registerRoute = $event->sender->docroute;

    	$isRegistered = $docRouteModel->docRouteExists($registerRoute,false);

        // only register new path
        if (!$isRegistered)
        {
            $model = new DocRoute();
            $model->title = $event->sender->title;
            $model->docroute = $registerRoute;
            $model->language = $event->sender->language;
            $model->setAttributes(array(
                    'itemModelClass' => $this->getModelClass(),
                    'itemCollectionName' => $this->getCollectionName(),
                    ),
                false);
            // set to 0 is important -> is no datetimestring
            $model->validfrom = 0;
            $model->validto = 0; //is set to max in beforesave
            $model->detachEventHandler('onAfterSave', array($route, 'registerDocRoute'));

			$model->save();

            if ($model->hasErrors()) {
                echo CHtml::tag('h1', array(), __METHOD__);
                die(CHtml::errorSummary($model));
            }

            return true;
        }

        return false;
    }

    /**
     * Define the indexes for this collection
     */
    public function indexes()
    {
        return array(

            'idx_docroute' => array(
                'key' => array(
                    'docroute' => EMongoCriteria::SORT_ASC,
                    ),
                ),

	        'idx_docroutid' => array(
	        	'key' => array(
	        	    'docrouteid' => EMongoCriteria::SORT_ASC,
	        	    ),
	        ),

        	'idx_contenttype' => array(
	            'key' => array(
	                'modelclass' => EMongoCriteria::SORT_ASC,
	                'docroute' => EMongoCriteria::SORT_ASC,
	              ),
	            ),

            'idx_valid' => array(
                'key' => array(
                    'status' => EMongoCriteria::SORT_ASC,
                    'validfrom' => EMongoCriteria::SORT_ASC,
                    'validto' => EMongoCriteria::SORT_ASC,
                    ),
                ),

	        'idx_tags' => array(
	            'key' => array(
	                'tags' => EMongoCriteria::SORT_ASC,
	                ),
	            ),

	        'idx_outputorder' => array(
	            'key' => array(
	                'weight' => EMongoCriteria::SORT_ASC,
	                'modified' => EMongoCriteria::SORT_ASC,
	                ),
	            ),
            );
    }

    /**
     *
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // NOTE: you should only define rules for those attributes that
        // will receive user inputs.
        return array(
            array('title', 'required'),
            array('language', 'length', 'max' => 20),
            array('status, weight', 'numerical', 'integerOnly' => true),
            array('title, subtitle, docroute', 'length', 'max' => 255),
            array('body', 'length', 'max' => 500000),
            array('attachments,data,tags', 'safe'),
            array('permissions,settings,externallinks', 'safe', 'on' => 'insert, update'),

            // array('attachment','file','allowEmpty' => true,'types' => 'zip, jpg, gif, png, pdf, doc, txt'),
            // todo: Problems with datetimeFormat
            // array('validfrom, validto', 'type', 'type'=>'datetime'),
            // 'datetimeFormat' => Yii::app()->controller->module->getValidationRuleDateTimeFormat()),
            array('validfrom, validto', 'safe'),
            array('validfrom, validto', 'unsafe', 'on' => 'search'),
            array('_id,title,body,author,docroute,status', 'safe', 'on' => 'search'),
             array(self::DOCROUTE_PREFIX_ATTR, 'safe', 'on' => 'insert, update',),
            );
    }

    /**
     *
     * @return array customized attribute labels (name=>label)
     */
    public function attributeLabels()
    {
        return array(
            'language' => MongoCmsModule::t('Langugage'),
            'weight' => MongoCmsModule::t('Weight'),
            'docroute' => MongoCmsModule::t('Route'),
            'title' => MongoCmsModule::t('Title'),
            'body' => MongoCmsModule::t('Body'),
            'tags' => MongoCmsModule::t('Tags'),
            'status' => MongoCmsModule::t('Status'),
            'created' => MongoCmsModule::t('Created'),
            'modified' => MongoCmsModule::t('Modified'),
            'validfrom' => MongoCmsModule::t('Valid from'),
            'validto' => MongoCmsModule::t('Valid to'),
            'author' => MongoCmsModule::t('Author'),
            'owner' => MongoCmsModule::t('Owner'),
            'modelclass' => MongoCmsModule::t('Class'),
            'attachments' => MongoCmsModule::t('Attachments'),
            'permissions' => MongoCmsModule::t('Permissions'),
            'externallinks' => MongoCmsModule::t('External links'),
            'data' => MongoCmsModule::t('Data'),
            );
    }
}