<?php

/**
 * Menu.php
 *
 * The model for menu building based on the model DocRoute
 *
 * Provides static functions for creating menu items
 * @see config/mongocms/createmenuitems_main.php
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class Menu extends DocRoute
{
	const MAIN = 'main'; //the docrouteid of the mainmenu

	/**
	 * Returns the static model of the specified AR class.
	 *
	 * @return the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}


	/**
	 * Enable setting permissions for Menu
	 *
	 * @return boolean
	 */
	public function exposePermissions()
	{
		return true;
	}


	/**
	 * Set docrouteid required
	 *
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// don't forget to merge with parent rules
		return array_merge(parent::rules(), array(
		            array('docrouteid', 'required'),
		            array('docrouteid', 'ext.YiiMongoDbSuite.extra.EMongoUniqueValidator','on' => 'insert'),
		        ));
	}


	/**
	 * If docroute exists
	 * - update docrouteid and set modelclass of the docroute record to Menu
	 *
	 * @return boolean
	 */
	public function beforeSave()
	{
		if (!parent::beforeSave())
			return false;

		if($this->getIsNewRecord())
		{
			if (Page::model()->docRouteExists($this->docroute,false))
			{
				//atomic update: assign the docrouteid and modelclass
				$values = array(
				           'docrouteid' => (string)$this->docrouteid,
				           'modelclass' => (string)$this->getModelClass(),
				           'title' => (string)$this->title,
				          );

				$criteria = array('docroute' => (string)$this->docroute);
				$this->atomicUpdate($criteria,$values);

				//redirect to manage menu
				$controller = Yii::app()->controller;
				$controller->redirect($controller->createUrl('/mongocms/menu'));
				return false;
			};
		}

		return true;
	}


	/**
	 *
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array_merge(parent::attributeLabels(),
		array(
				'title' => MongoCmsModule::t('Menu caption'),
				'docrouteid' => MongoCmsModule::t('Menu ID'),
			));
	}

	/**
	 * Menu item for login
	 *
	 * @param string $label
	 * @return
	 */
	public function getLoginItem($label='Login')
	{
		$url = Yii::app()->mongocmsModule()->loginUrl;
		return array('label'=>MongoCmsModule::t($label), 'url'=>$url, 'visible'=>Yii::app()->user->isGuest);
	}

	/**
	 * Menu item for logout
	 *
	 * @param string $label
	 * @return
	 */
	public function getLogoutItem($label='Logout')
	{
		$url = Yii::app()->mongocmsModule()->logoutUrl;
		return array('label'=>MongoCmsModule::t($label), 'url'=>$url, 'visible'=>!Yii::app()->user->isGuest);
	}

	/**
	 * Add all root items to a menu
	 * Means: all docroutid starting with $docrouteIdPrefix
	 *
	 * @see config/mongocms/createmenuitems_main.php
	 *
	 * @param string $docrouteIdPrefix
	 * @return array
	 */
	public function addItems(&$menuItems,$docrouteId=null)
	{
		$collection = Yii::app()->mongocmsCollection($this->getCollectionName());
		if (!isset($docrouteId))
		   $criteria = array('docrouteid' => $docrouteId);

		if (!empty($docrouteId))
		{
			if (is_string($docrouteId))
				$criteria = array('docrouteid' => $docrouteId);
			elseif (is_array($docrouteId))
				$criteria = array('docrouteid' => array('$in' => $docrouteId));
		}
		else
			$criteria = array('docrouteid' => array('$ne' => null));

		$cursor = $collection->find($criteria, array('docrouteid'));

		if (!empty($cursor) && $cursor->count())
			foreach ($cursor as $id => $value)
			{
				$items = $this->getMenuItems($value['docrouteid'],Page::model()->getCollectionName());
				if (!empty($items))
				  $menuItems[] = $items; //add the first item as root item
			}
	}

	/**
	 * Return the menu items to manage content for all contenttypes
	 * Check access to the contenttype
	 *
	 * @return array
	 */
	public static function getManageItems($label,$excludeContenttypes=array(),$includeContenttypes=array())
	{
		$items = array();
		if (Yii::app()->user->isGuest)
			return $items;

		$contentTypes = Yii::app()->mongocmsContentTypes();

		foreach ($contentTypes as $className => $description)
		{
			if (in_array($className,$excludeContenttypes))
				continue;

			if (empty($includeContenttypes) || in_array($className,$includeContenttypes))
			{

				$module = Yii::app()->mongocmsModuleFromContenttype($className);
				$route = Page::model($className)->getCRUDControllerRoute(null,true,$module->id);
				$visible = Page::model($className)->checkContenttypeAccess('manage',array(),true);
				$url = Yii::app()->createUrl('/'.$route . '/admin?modelClass='.$className);
				$items[] = array(
								'label'=>MongoCmsModule::t($description),
								'url'=>$url,
								'visible'=>$visible,
							);
			}
		}

		if (DocRoute::checkParentItemVisible($items)) //check if has children with access allowed
		    return array('label'=>MongoCmsModule::t($label), 'url'=>array('#'),'items'=>$items);
		else
	        return array();
	}


	/**
	 * Get the menu item for cms action of the admin controller
	 * Check access for global defined permissions in mongocms
	 * @see config/mongocms/roleoperations_mongocms.php
	 *
	 * @param string $label
	 * @return array
	 */
	public static function getAdminCmsItem($action,$checkOperation,$label)
	{
		$item = array();

		if (Yii::app()->mongocmsCheckAccess($checkOperation, array(),true))
		{
			$route = Yii::app()->mongocmsControllerRoute('admin',false);
			$url = Yii::app()->createUrl($route . '/' .$action);
			$item = array(
	                      'label'=>MongoCmsModule::t($label),
	                      'url'=>$url,
	                     );
		}

		return $item;
	}


	/**
	 * Get the url for a controller action
	 * Checks if controller is mapped
	 *
	 * @param string $controllerId
	 * @param string $route
	 * @param array $urlParams
	 * @return
	 */
	public static function getControllerActionUrl($controllerId,$route='',$urlParams = array())
	{
		$controllerRoute = Yii::app()->mongocmsControllerRoute($controllerId);
		$route = empty($route) ? $controllerRoute : $controllerRoute . '/' . $route;
		return Yii::app()->createUrl($route, $urlParams);
	}


	/**
	 * The menu item for a controller action
	 *
	 * @param string $label
	 * @param array $urlParams
	 * @return array
	 */
	public static function getControllerActionItem($controllerId,$route,$label,$urlParams = array())
	{
		$url = self::getControllerActionUrl($controllerId,$route,$urlParams);
		return array('label'=>MongoCmsModule::t($label),'url'=>$url);
	}



	/**
	 * The menu item for a controllers default action
	 *
	 * @see config/mongocms/createmenuitems_main.php
	 *
	 * @param string $label
	 * @param array $urlParams
	 * @return array
	 */
	public static function getControllerDefaultActionItem($controllerId,$label,$urlParams = array())
	{
	   return self::getControllerActionItem($controllerId,'',$label,$urlParams);
	}


	/**
	 * The menu item for a model action
	 *
	 * @param mixed $model name of modelclass or object
	 * @param string $action the action: update, delete ....
	 * @param mixed $label
	 * @param mixed $controllerId
	 * @param array $urlParams
	 * @return
	 */
	public static function getActionItem($model,$action, $id = null, $label = null,$controllerId = null,$subModuleId='',$urlParams = array())
	{
        $item = array();

		if (is_string($model))
        	$model = Page::model($model);

		if (isset($id))
			$urlParams = array_merge($urlParams,array('id'=>$id));

		$route = $model->getCRUDControllerRoute($controllerId,false,$subModuleId);
		$url = '/'.$route .'/' .$action;


		//problems with normalizeUrl, Yii::app()->createUrl($url,$urlParams) and id param ...
		//create manually
		if (!empty($urlParams))
		{
			$params = '';
			foreach ($urlParams as $key=>$value)
				$params = empty($params) ? "?$key=$value" : "&$key=$value";
			$url .= $params;
		}

		if (!empty($url))
		{
			if (!isset($label))
				$label = MongoCmsModule::t($model->getModelClass()) . ' ' . MongoCmsModule::t($action);

			//$urlArray = array_merge(array($url),$urlParams);
			//var_dump($urlArray);
			//$url = CHtml::normalizeUrl($urlArray);
			$item = array('label'=>MongoCmsModule::t($label),'url'=>Yii::app()->createUrl($url));
		}

		return $item;
	}


	/**
	 * The url to a specific referenced content of the current user
	 *
	 * @param string $referencType @see model Reference.php
	 * @param array $urlParams
	 * @return
	 */
	public static function getUserReferencesUrl($referencType,$urlParams = array())
	{
		if (Yii::app()->user->isGuest)
			return '';

		$route = Yii::app()->mongocmsControllerRoute('user');
		$urlParams = array_merge($urlParams,array('action'=>'list','type'=>$referencType));
		return Yii::app()->createUrl($route .'/references',$urlParams);
	}

	/**
	 * The menu item for referenced content of the current user
	 *
	 * @param string $referencType @see model Reference.php
	 * @param string $label
	 * @param string $controllerId
	 * @param array $urlParams
	 * @return
	 */
	public static function getUserReferencesListItem($referencType,$label,$urlParams = array())
	{
	    return empty($url) ? array() : array('label'=>MongoCmsModule::t($label),'url'=>$url,);
	}


	/**
	 * Returns the url of the action 'tag' of the ContentController
	 *
	 * @param string $tag
	 * @param array $urlParams
	 * @return string
	 */
	public static function getTagUrl($tag,$urlParams = array())
	{
		$urlParams = array_merge($urlParams,array('tag'=>$tag));
		return self::getControllerActionUrl('content','tag',$urlParams);
	}

	/**
	 * Returns the item with the action 'tag' of the ContentController
	 *
	 * @param string $tag
	 * @param array $urlParams
	 * @return string
	 */
	public static function getTagItem($tag,$label,$urlParams = array())
	{
		$url = self::getTagUrl($tag,$urlParams);
		return array('label'=>MongoCmsModule::t($label),'url'=>$url);
	}

}
