<?php

/**
 * DocRoute.php
 *
 * The base model Menu
 * Handling of checking access to docroutes and building menus
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class DocRoute extends Page
{
	public $itemCollectionName; //the collectionname of referenced item: mongocms_content or user
    public $docrouteid; //the id of the menu route item

	protected static $_checkedRoutes;

    /**
     * Assign configured mongodb from module
     * Assign modelclass, ... and SoftAtrributes
     *
     * @param string $scenario
     */
    public function __construct($scenario = 'insert')
    {
        parent::__construct($scenario);
        $this->attachEventHandler('onBeforeDelete', array($this, 'docrouteBeforeDelete'));
    }

	/**
	 * The mongo collection where this document is stored
	 * @return string
	 */
	public function getCollectionName()
	{
		return 'mongocms_docroute';
	}


    /**
     * Returns the static model of the specified AR class.
     *
     * @return the static model class
     */
    public static function model($className = __CLASS__)
    {
        return parent::model($className);
    }

	/**
	 * Override from parent
	 * No page settings for user supported
	 *
	 * @return CForm
	 */
	public function createSettingsForm()
	{
		return null;
	}

	/**
	 *
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array_merge(parent::attributeLabels(),
         array(
		    'itemCollectionName' => MongoCmsModule::t('Collection'),
		    'docrouteid' => MongoCmsModule::t('ID'),
		    ));
	}


    /**
     * Override getVisibleAttributes() from Page
     *
     * @param mixed $actionId
     * @return
     */
    public function getVisibleAttributes($actionId = null)
    {
        if (!isset($action))
            $actionId = Yii::app()->controller->action->id;

        switch ($actionId) {
            case 'admin':
                return array('docrouteid', 'docroute', 'title', 'status','itemCollectionName');
                break;

            default:
                return array('docrouteid', 'docroute', 'status', 'author', 'itemCollectionName',
                    'created', 'modified', 'validfrom', 'validto', 'title', 'body');
        }
    }

	/**
	 * Define the indexes for this collection
	 */
	public function indexes()
	{
		return array(

		'idx_docroute' => array(
		        'key' => array(
		            'docroute' => EMongoCriteria::SORT_ASC,
		            ),
		         'unique'=>true,
		        ),

		'idx_docroutid' => array(
			'key' => array(
			    'docrouteid' => EMongoCriteria::SORT_ASC,
			    ),
		),

		'idx_contenttype' => array(
			'key' => array(
			    'modelclass' => EMongoCriteria::SORT_ASC,
			    'docroute' => EMongoCriteria::SORT_ASC,
			  ),
		),

	    'idx_published' => array(
	        'key' => array(
	            'status' => EMongoCriteria::SORT_ASC,
	            ),
	        ),
	    );
	}

    /**
     *
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        // don't forget to merge with parent rules
        return array_merge(parent::rules(), array(
                //array('docroute', 'ext.YiiMongoDbSuite.extra.EMongoUniqueValidator','on' => 'insert'),
                array('docrouteid,docroute', 'unsafe','on' => 'update'),
                array('docroute', 'required'),
                array('docroute', 'ext.YiiMongoDbSuite.extra.EMongoUniqueValidator','on' => 'insert'),
                array('itemCollectionName', 'safe','on' => 'search'),
                ));
    }

	/**
	 * Ensure docrouteid is null if empty
	 * Set itemCollectionName to 'mongocms_content' if empty
	 *
	 * @return boolean
	 */
	public function beforeSave()
	{
		if (!parent::beforeSave())
			return false;

		if (empty($this->docrouteid))
			$this->docrouteid = null;

		if (empty($this->itemCollectionName))
			$this->setAttributes(array(
			   'itemCollectionName' => Page::model()->getCollectionName(),
			),
		   false);

		return true;
	}

    /**
     * Dont allow delete if a content exists with this docroute
     * If is Menu don't delete
     *
     * @param mixed $event
     * @return
     */
    public function docrouteBeforeDelete($event)
    {
        $hasDoc = Page::model()->docRouteExists($event->sender->docroute,false);

    	//delete menu: only reset docrouteid and change modelclass
		if ($this instanceof Menu)
    	{
    		$values = array('docrouteid' => null,
    	                    'modelclass' => 'DocRoute');
    		$criteria = array('_id' => new MongoId($this->_id));
    		$this->atomicUpdate($criteria,$values);

			if ($hasDoc)
			{
				if (Yii::app()->request->isAjaxRequest)
					die(); // workaround for CGridView
				else
					return false;
			}
    	}

        if (!$hasDoc)
            return true;
        else
	        if (Yii::app()->request->isAjaxRequest) // workaround for CGridView
	            throw new CHttpException(400, MongoCmsModule::t('Cannot delete this record.'));
	        else
	            return false;
    }

	/**
	 * Helper function to find the intersection of 2 routes
	 * Usage for finding selected for dropdowns
	 * @see MongoCmsHelper.docrouteFormElements
	 *
	 * @param mixed $route1
	 * @param mixed $route2
	 * @return
	 */
	public static function getIntersectRoute($route1,$route2)
	{
		$arr1 = explode('/',$route1);
		$arr2 = explode('/',$route2);

		$intersect = array_intersect($arr1,$arr2);

		return empty($intersect) ? '' : implode('/',$intersect);
	}


	/**
	 * Check 'access' permission to a docroute and it's subparts
	 * Uses self::$_checkedRoutes for memory caching
	 *
	 * for example docroute = /x/y/z
	 * check /x, /x/y, /x/y/z
	 *
	 * @param string $docRoute
	 * @param boolean $checkOnly
	 * @return
	 */
	public function checkDocRouteAccess($docRoute,$checkOnly = false)
	{
		if (empty($docRoute))
			return false;

		if (!isset(self::$_checkedRoutes))
			self::$_checkedRoutes = array();

		if (isset(self::$_checkedRoutes[$docRoute]))
			return self::$_checkedRoutes[$docRoute];

		//if ($this->checkCachedRoute($docRoute,$isAllowed))
		//	return $checkOnly ? $isAllowed : $this->accessDenied();

		$parts = preg_split('/\//', $docRoute);
		$firstPart = $parts[0];

		//find all subparts at once, don't request mongodb for each part
		$regexObj = new MongoRegex("/^$firstPart/");
		$criteria = array(
		              'docroute' => $regexObj,
		              'docroute' => array('$lte' => $docRoute),
					 );

		$collection = Yii::app()->mongocmsCollection($this->getCollectionName());

		//only load 'docroute', 'permissions'
		$cursor = $collection->find($criteria, array('docroute', 'permissions'));

		if (!empty($cursor) && $cursor->count())
		{
			$cursor->sort(array('docroute' => EMongoCriteria::SORT_ASC));

			foreach ($cursor as $id => $value)
			{
				if (strpos($docRoute,$value['docroute']) !== false)
				{
					if (isset(self::$_checkedRoutes[$docRoute]))
					{
						if (!self::$_checkedRoutes[$docRoute])
							return false;
						else
							continue;
					}

					$model = new DocRoute;
					$model->docroute = $value['docroute'];
					$model->permissions = $value['permissions'];
					$access = $model->checkContentAccess('access', $checkOnly);

					self::$_checkedRoutes[$docRoute] = $access;

					if (!$access)
						return false;
				}
			}
		}

		return true;
	}

    /**
     * Find all Ids from a collection with overall checkAccess
     * If operations is an array, all operations have to be allowed
     *
     * @param string $collectionName
     * @param string $collectionName
     * @param mixed $route
     * @param boolean $publishedOnly
     * @param boolean $returnOnFirstFound : don't return all found content
     * @return array
     */
    public function findAllowedDocumentIds(
									      $rootDocRoute = null,
    	 								  $additionalCriteria = null,
									      $publishedOnly = true,
									      $returnOnFirstFound = false,
    	                                  $collectionName = 'mongocms_content',
									      $operations = 'view',
									      $limit = null
    	                                 )
    {
        $allowedContent = array();
        $isRootUser = Yii::app()->mongocmsIsRootUser();
        $roles = Yii::app()->mongocmsCurrentUserRoles();

        if (is_string($operations))
            $operations = array($operations);

        $collection = Yii::app()->mongocmsCollection($collectionName);
        $criteria = array();

        // add criteria for publishedOnly
		if ($publishedOnly)
        {
            $time = time();
			$criteria = array_merge($criteria,
                array(
                    'status' => array('$gt' => 0),
                    'validfrom' => array('$lte' => $time),
                    'validto' => array('$gte' => $time),
                    )
                );
        }

        // find all where docroute starting with $rootDocRoute
        if (isset($rootDocRoute))
        {
            $criteria = array_merge($criteria,
                array(
                       'docroute' => array('$regex' => "^$rootDocRoute"),
                    )
                );
        }

    	// add additional criteria
    	if (is_array($additionalCriteria))
			foreach ($additionalCriteria as $key => $value)
	    		$criteria[$key] = $value;

        // load only _id,docroute, modelclass and permissions
        $cursor = $collection->find($criteria, array('_id', 'docroute', 'modelclass', 'permissions'));

    	$emptyPermissions = array();
    	$contentPermissions = array();

    	// --------------- check content access -----------------------
		if (!empty($cursor) && $cursor->count())
    	{

			if (isset($limit))
    			$cursor->limit($limit);

			$cursor->sort(array('docroute' => EMongoCriteria::SORT_ASC));

    		foreach ($cursor as $id => $value) {
				if ($isRootUser)  // all content is allowed
    				$allowedContent[] = $value;
    			else
    			{

					//check docroute access first
    				if (!$this->checkDocRouteAccess($value['docroute'],true))
    					return $allowedContent;

					if (empty($value['permissions'])) // no content permissions assigned
    					$emptyPermissions[] = $value;
    				else
    				{
	   					$allowed = 0;
	   					foreach ($operations as $operation) {
	   						$allowedRoles = array_intersect(array_keys($roles),
	   							               $value['permissions']['roles'][$operation]);
	   						if (!empty($allowedRoles))
	   							$allowed ++;
	   					}

	   					if (count($operations) == $allowed)
	   						$allowedContent[] = $value; //all operations are allowed
    				}
    			}
    		}
    	}


        if (empty($emptyPermissions) && empty($allowedContent))
            return array(); //no allowed content found

        if (empty($emptyPermissions) && !empty($allowedContent))
            return $allowedContent; //only allowed found, no need to check contenttypes

        if (!empty($allowedContent) && $returnOnFirstFound)
            return $allowedContent; //allowed content found, no more checks for contenttype permissions


        // ---- check contenttype access for items with empty permissions -------
        $contentTypeAuthManager = new MongoCmsAuthManager('');
        $collection = $contentTypeAuthManager->getCollection();

    	$inArray = array();
    	foreach ($emptyPermissions as $id => $value)
    		$inArray[] = $value['modelclass'];

        $criteria = array(
              'authId' => array('$in' => $inArray),
            );

    	foreach ($operations as $operation)
    		$criteria[$operation] = array('$exists' => true);

        $cursor = $collection->find($criteria);

        if (!empty($cursor) && $cursor->count())
        {
			foreach ($roles as $role => $label) {
                foreach ($cursor as $id => $value)
                {
                    if (isset($value[$role]) && $value[$role]['type'] == 2 && !empty($value[$role]['children']) &&
                            in_array($operation, $value[$role]['children']))
                    {

						// add the items of the allowed contenttype to allowed
						foreach ($emptyPermissions as $id => $item)
	                        if ($item['modelclass'] == $value['authId'])
	                            $allowedContent[] = $item;

                        return $allowedContent;
                    }
                }
            }
        }

        return array(); //empty
    }

	/**
	 * Return all allowed docroutes
	 *
	 * @param string $rootDocRoute
	 * @param string $docRouteId
	 * @param mixed $collectionName
	 * @param mixed $modelClassOnly
	 * @param boolean $publishedOnly
	 * @param int $limit
	 * @param array $attributes
	 * @return MongoCursor
	 */
	protected function findAllowedDocRoutes($rootDocRoute,
	                                        $docRouteId = null,
	                                        $modelClassOnly = false,
	                                        $collectionName = null,
	                                        $publishedOnly = false,
											$limit = null,
		                                    $operations = 'access',
											$attributes = array()
											)
	{
		if (!isset($collectionName))
			$collectionName = $this->getCollectionName();

		$criteria = array();

		if ($modelClassOnly)
			$criteria['modelclass'] = $this->getModelClass();

		if (isset($docRouteId))
			$criteria['docrouteid'] = $docRouteId;

		$allowedItems = $this->findAllowedDocumentIds($rootDocRoute,
			                                         $criteria,
			 										 $publishedOnly,
													 false, //$returnOnFirstFound,
													 $collectionName,
					                                 $operations,
					                                 $docRouteId,
					                                 $limit);

		if (empty($allowedItems))
			return array();

		foreach ($allowedItems as $item)
			$allowedIds[] = new MongoID($item['_id']);


		$collection = Yii::app()->mongocmsCollection($collectionName);
		$criteria = array(
		   '_id' => array('$in' => $allowedIds),
		);

		$attributes = array_merge($attributes,array('_id','docrouteid','docroute','title','permissions','status'));

		$cursor = $collection->find($criteria,$attributes);

		if (!empty($cursor))
			$cursor->sort(array('docroute' => EMongoCriteria::SORT_ASC));

		return $cursor;
	}


	/**
	 * Get the info  from a specific docrouteid
	 *
	 * @param string $menuId
	 * @return array
	 */
	protected function findDocRouteId($docRouteId,$modelClassOnly = false,$attributes = array())
	{
		$collection = Yii::app()->mongocmsCollection($this->getCollectionName());
		$criteria = array('docrouteid' => $docRouteId);

		if ($modelClassOnly)
			$criteria['modelclass'] = $this->getModelClass();

		$attributes = array_merge($attributes,array('_id', 'docrouteid', 'docroute', 'title'));

		$cursor = $collection->findOne($criteria, $attributes);

		return empty($cursor) ? false : $cursor;
	}


	/**
	 * Find all docroutes with docrouteid = null
	 * with the given itemCollectionName
	 *
	 * @param string $itemCollectionName
	 * @return
	 */
	public function getDocRoutesOnly($itemCollectionName = 'mongocms_content', $attributes = array())
	{
		$collectionName = $this->getCollectionName();

		$collection = Yii::app()->mongocmsCollection($collectionName);
		$criteria = array('docrouteid' => null,'itemCollectionName'=>$itemCollectionName);

		$attributes = array_merge($attributes,array('_id', 'docroute', 'title'));

		$cursor = $collection->find($criteria, $attributes);
		return empty($cursor) ? false : $cursor;
	}

	/**
	 * DocRoute::getDocRouteIds()
	 *
	 * @param string $docRouteId
	 * @return
	 */
	public function getDocRouteIds($attributes = array())
	{
		return $this->findAllowedDocRoutes(null,null,true,null,false,null,'access',$attributes);
	}

	/**
	 * DocRoute::getDocRoutes()
	 *
	 * @param string $docRouteId
	 * @return
	 */
	public function getDocRoutes($docRouteId,$modelClassOnly = true, $attributes = array())
	{
		$cursor = $this->findDocRouteId($docRouteId,$modelClassOnly, $attributes);

		if ($cursor === false)
			return array();

		return $this->findAllowedDocRoutes($cursor['docroute']);
	}

	/**
	 *
	 * First check 'access' access to docroute
	 * Check 'view' access to "content behind"
	 *
	 * @param mixed $docRoute : single record from mongocms_content
	 * @param mixed $allowedContentExists : returns true if content exists behind an item
	 * @return boolean
	 */
	protected function checkCollectionDocRouteAccess($collectionName, $docRoute, &$allowedContentExists)
	{
		$allowedContentExists = false;

		// check access to the docroute
		if (!$this->checkDocRouteAccess($docRoute,true))
			return false;

		// if access to docroute allowed,
		// check at least 'view' access to at least one content of this path
		$criteria = array('docroute'=>$docRoute);
		$allowedItems = $this->findAllowedDocumentIds(null, //$rootDocRoute,
			                                         $criteria,
			 										 true, //$publishedOnly,
													 true, //$returnOnFirstFound,
													 $collectionName,
					                                 'view', //$operations,
					                                 null, //$docRouteId,
					                                 null //$limit
					                                 );

		$allowedContentExists = !empty($allowedItems);
		return true;
	}



	/**
	 * Returns menu items with published content and access for the user allowed
	 * Prepared for CMenu or other ...
	 *
	 * @param mixed $docRoutes
	 * @param string $clickPath : onclick path (corresponding with controller action)
	 * @return
	 */
	protected function buildMenuItems($collectionName,$cursor, $clickPath,&$firstDocRoute)
	{
		$tree = array();
		$regPath = array();

		$firstDocRoute = null;

		foreach ($cursor as $id => $value)
		{
			if ($this->checkCollectionDocRouteAccess($collectionName,$value['docroute'],$allowedContentExists))
			{
			   if (!isset($firstDocRoute))
			   	 $firstDocRoute = $value['docroute'];

			   $parts = preg_split('/\//', $value['docroute']);

			   $curPath = empty($parts) ? '' : $parts[0];
			   $lastPart = array_pop($parts);


			   $parent = &$tree;

			   foreach ($parts  as $part)
			   {
					$item = array(
							'label' => $part,
						   	'url' => array('#'),
						   	'items' => array(),
						   );
				   	$idx = count($parent);

					$isNewReg = true;
					if (array_key_exists($curPath,$regPath))
					{
				   			$parent = &$regPath[$curPath];
				   		    $idx = 0;
						    $curPath = '';
						    $isNewReg = false;
				   	}
				   	else
				   	{
				   	    $parent[] = $item;
				   		//$regPath[$curPath] = &$parent;
				   	}


                   //if (!empty($curPath))
			      $regPath[$curPath] = &$parent;

			   	  $parent = &$parent[$idx]['items'];

			   	  $curPath .= $isNewReg ? $part : '/' . $part;

			   }

				// add last item
				if (!count($parent['items']))
				{
					$item = array(
								'label' => $value['title'],
								'url' => !$allowedContentExists
								? '#'
								: Yii::app()->createUrl($clickPath . '/' . $value['docroute']),
							);
					$parent[] = $item;
					$regPath[$curPath] = &$parent;
				}
			}
		}

		return $tree;
	}


	/**
	 * Return the checked menuitems of a specific docrouteid
	 *
	 * @param string $DocRouteId
	 * @param string $collectionName
	 * @return array
	 */
	public function getMenuItems($docrouteId,$collectionName = Page::COLLECTION_NAME)
	{
		//find docrouteid: uses findOne -> can access cursor results directly as array
		$cursorDocRouteId = $this->findDocRouteId($docrouteId,true);

		if ($cursorDocRouteId === false)
			return array();


		$rootDocRoute = $cursorDocRouteId['docroute']; //the docroute at the menu entrypoint
		$menuTitle = $cursorDocRouteId['title']; //the title of the menu

		$cursorAllowed = $this->findAllowedDocRoutes($rootDocRoute,
		                              null,//$docRouteId
		                              false, //$modelClassOnly
		                              $collectionName,
		                              true, //$publishedOnly
		                              null, //$limit
		                              'view' //$operations
		                              );

		if (!empty($cursorAllowed) && $cursorAllowed->count())
		{
			//actionPage of the ContentController
			$clickPath = Yii::app()->mongocmsControllerRoute('content') .'/page';
			$items = $this->buildMenuItems($collectionName,$cursorAllowed,$clickPath,$firstDocRoute);

			if (!empty($items))
			{
				if ($firstDocRoute == $rootDocRoute)
			   	 //There is a content behind the menuroot
			   	 //Set the first item as menu root
			   	return $items[0];
			   else
			   	//generate an extra menuroot with the menu title as caption
			    return array('label'=>MongoCmsModule::t($menuTitle), 'url'=>'#', 'items'=>$items);
			}
		}

		return array();
	}


    /**
     * Returns the available operations for checking contentaccess
     * These operations can be added to permissions
     *
     * @return array
     */
    public function getContentAccessOperations()
    {
        return array('access' => 'Access menu');
    }

    /**
     * named scope: byItemCollectionName
     */
    public function byItemCollectionName($name)
    {
        $criteria = $this->getDbCriteria();
        $criteria->itemCollectionName = $name;
        $this->setDbCriteria($criteria);

        return $this;
    }


	/**
	 * Get all valid docroutes from content starting with rootDocRoot
	 * valid: status>0, validfrom/to
	 * Only load _id, title, docroute and permissions
	 *
	 * @param string $menu
	 * @return EMongoCriteria
	 */
	public function getValidContentDocRoutes($rootDocRoot)
	{
		$time = time();
		$criteria = $this->getDbCriteria();
		$criteria->itemCollectionName = 'mongocms_content';
		// starting with $rootDocRoot
		$criteria->docroute = new MongoRegex("/^$rootDocRoute/");
		$criteria->status('>', 0);
		$criteria->validfrom('<=', $time);
		$criteria->validto('>=', $time);
		$criteria->sort('docroute', EMongoCriteria::SORT_ASC);
		$criteria->select(array('_id', 'title', 'docroute', 'permissions'));
		$this->setDbCriteria($criteria);

		return $this;
	}

    /**
     * Used for building mainmenu
     *
     * @see config/mongocms/menuitems_main.php
     * @param array $subitems
     * @return boolean
     */
    public static function checkParentItemVisible($subitems)
    {
        if (empty($subitems))
            return false;

        foreach ($subitems as $subitem)
        if (!isset($subitem['visible']) || $subitem['visible'])
            return true;
        return false;
    }
}
