<?php

/**
 * AdminController.php
 *
 * This is the defaultcontroller for the mongocms
 * It is the base for other controllers (MenuController, UserController ...)
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */

Yii::import('mongocms.models.*');
Yii::import('mongocms.components.MongoCmsController');

class AdminController extends MongoCmsController {
    public $defaultAction = 'admin';
    // set layout to mongocms/views/layout
    // so we can design a different layout for cms in our application
    public $layout = 'column1';



	/**
	 * Have to assign the module in __construct
	 * if loaded directly from controllerMap
	 *
	 * @param string $id
	 * @param string $module
	 */
	public function __construct($id, $module = null)
	{
		if (!isset($module))
			$module = Yii::app()->mongocmsModule();

		parent::__construct($id, $module);
	}


	/**
	 * Validate external links submitted by the form
	 * and add non empty items to the array $externalLinks
	 *
	 * @param ExternalLink $linkModel
	 * @param array $externalLinks
	 * @return boolean
	 */
	protected function validateExternalLinks(&$linkModel,&$externalLinks)
	{
		$externalLinks = array();
		$linksValid = true;

		if (empty($_POST['ExternalLinks']) || empty($_POST['ExternalLinks']['url']))
			return $linksValid;

		$idx = 0;
		foreach ($_POST['ExternalLinks']['url'] as $url)
		{
			$url = trim($url);
			$label = trim( $_POST['ExternalLinks']['label'][$idx]);

			if (!empty($url))
			{
				$linkModel = new ExternalLink;

				$linkModel->url = $url;
				$linkModel->label = $label;

				if ($linkModel->validate())
					$externalLinks[$url] = $label;
				else {
					$linksValid = false;
					break;
				}
			}
			$idx++;
		}

		return $linksValid;
	}

    /**
     * Displays a particular model.
     *
     * @param integer $id the ID of the model to be displayed
     */
    public function actionView($id)
    {
        $model = $this->loadModel($id);
    	$model->checkAccess('view');

    	if (Yii::app()->request->isAjaxRequest)
          $this->renderPartial($model->getView('view'), array('model' => $model));
    	else
          $this->render($model->getView('view'), array('model' => $model));
    }

    /**
     * Creates a new model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     */
    public function actionCreate($modelClass = null, $docroute = null)
    {
        if (!isset($modelClass))
            $modelClass = 'Page';

        $model = new $modelClass();
    	$model->checkContenttypeAccess('create');

    	$linkModel = new ExternalLink;

    	if (!empty($docroute)) {
    		DocRoute::model()->checkDocRouteAccess($docroute);
    		$model->staticDocRoute = $docroute;
    	}

        if (isset($_POST[$modelClass])) {

            $attributes = $_POST[$modelClass];
            // add the values from the embedded values of the form
            foreach ($model->embeddedDocuments() as $doc => $embModelClass) {
                if (isset($_POST[$embModelClass])) {
                    $attributes[$doc] = $_POST[$embModelClass];
                }
            }

            $model->attributes = $attributes;
            if (!empty($model->softAttributes)) {
                $softAttributes = $model->getSoftAttributeNames();

                if (!empty($softAttributes))
                    foreach ($softAttributes as $attribute)
                    $model->$attribute = $_POST[$modelClass][$attribute];
            }

        	//assign the settings from PageSettings
        	$model->settings = $_POST['PageSettings'];

        	//assign the links from PageLinks
        	if (($linksValid = $this->validateExternalLinks($linkModel,$externalLinks)) !== false)
        		$model->externallinks = $externalLinks;

            if ($linksValid && $model->save()) {
                $this->redirect(array('view', 'id' => $model->_id));
            }
        }

        $this->render($model->getView('edit'), array(
                'model' => $model,
                'linkModel' => $linkModel, //for errorSummary
                ));
    }

    /**
     * Updates a particular model.
     * If update is successful, the browser will be redirected to the 'view' page.
     *
     * @param integer $id the ID of the model to be updated
     */
    public function actionUpdate($id)
    {
        $model = $this->loadModel($id);
    	$linkModel = new ExternalLink;

    	//check operation update OR updateAny OR updateOwn ...
    	//have to add 'update' because of checking docroute 'update'!
    	$model->checkAccess(array('update','updateAny','updateSameRole','updateSameUsertype','updateOwn'),
    		                array('model'=>$model));

        $modelClass = $model->getModelClass();

        if (isset($_POST[$modelClass])) {
            $attributes = $_POST[$modelClass];

            // add the values from the embedded values of the form
            foreach ($model->embeddedDocuments() as $doc => $embModelClass) {
                if (isset($_POST[$embModelClass])) {
                    $attributes[$doc] = $_POST[$embModelClass];
                }
            }

			$model->attributes = $attributes;
        	//assign the settings from PageSettings
        	$model->settings = $_POST['PageSettings'];

        	//assign the links from PageLinks
        	if (($linksValid = $this->validateExternalLinks($linkModel,$externalLinks)) !== false)
        		$model->externallinks = $externalLinks;

            if ($linksValid && $model->save())
            {
                $this->redirect(array('admin'));
            	// $this->redirect(array('view', 'id' => $model->_id));
            }
        }

        $this->render($model->getView('edit'), array(
                'model' => $model,
                'linkModel' => $linkModel, //for errorSummary
                ));
    }

    /**
     * Deletes a particular model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     *
     * @param integer $id the ID of the model to be deleted
     */
    public function actionDelete($id)
    {
        if (Yii::app()->request->isPostRequest) {
            // we only allow deletion via POST request
            $model = $this->loadModel($id);

        	//check operation delete OR deleteAny OR deleteOwn ...
        	//have to add 'delete' because of checking docroute 'delete'!
        	$model->checkAccess(array('delete','deleteAny','deleteSameRole','deleteSameUsertype','deleteOwn'),
        		                array('model'=>$model));

            $model->delete();
            // if AJAX request (triggered by deletion via admin grid view), we should not redirect the browser
            if (!isset($_GET['ajax']))
                $this->redirect(isset($_POST['returnUrl']) ? $_POST['returnUrl'] : array('admin'));
        } else
            throw new CHttpException(400, 'Invalid request. Please do not repeat this request again.');
    }

    /**
     * Lists all models.
     */
    public function actionIndex($modelClass = null)
    {
        if (!isset($modelClass))
            $modelClass = 'Page';

        $dataProvider = new EMongoDocumentDataProvider($modelClass);
        $this->render('index', array(
                'dataProvider' => $dataProvider,
                ));
    }

    /**
     * Manages all models.
     * A specific modelClass can be filtered
     */
    public function actionAdmin($modelClass = null)
    {
		if (!isset($modelClass))
    		$modelClass = 'Page';

    	$model = new $modelClass('search');
    	$model->checkAccess('manage');

    	$model = $model->byContentType();

    	if (isset($_GET[$modelClass]))
    		$model->setAttributes($_GET[$modelClass]);

    	$this->render($model->getView('admin'), array(
    	        'model' => $model,
    	        ));
    }


    /**
     * Manages the permissions for contenttypes
     */
    public function actionPermissions()
    {
    	Yii::app()->mongocmsCheckAccess('managePermissions');

    	$cacheFlushed = false;
    	//save global cms permissions
		if (isset($_POST['CmsPermissions']))
    	{
			$authManager = Yii::app()->mongocmsAuthManager();
    		$authManager -> removeRoleOperations();

    		foreach ($_POST['CmsPermissions'] as $operation => $roles)
				foreach ($roles as $role)
    			  $authManager->addRoleOperations($role,array($operation));

    		$authManager->save();

			//flush menu cache
			Yii::app()->mongocmsMenuCache->flush();

			//flush page cache
			Yii::app()->mongocmsPageCache->flush();

			$cacheFlushed = true;
    	}

    	//save contenttype permissions
		if (isset($_POST['Permissions']))
		{
        	foreach ($_POST['Permissions'] as $contentType => $formdata)
        		Page::model($contentType)->saveContenttypePermissions($formdata);

			if (!$cacheFlushed)
			{
				//flush menu cache
				Yii::app()->mongocmsMenuCache->flush();

				//flush page cache
				Yii::app()->mongocmsPageCache->flush();
			}
        }
    	else
    		//install the contenttype permissions
    		Yii::app()->mongocmsModule()->install();

        $this->render('permissions');
    }


	/**
	 * Recreate all contenttype permissions from config files 'roleoperations_MODELCLASS'
	 * This method should be called if one of the config files has been changed
	 * This action is only allowed for root user
	 *
	 */
	public function actionRecreatePermissions()
	{
	   Yii::app()->mongocmsCheckAccess('recreatePermissions');

		if (Yii::app()->request->isAjaxRequest)
		{
			echo ' '; //IMPORTANT: exception message from scripts will only be display with an output before !?
			Yii::app()->mongocmsModule()->recreatePermissions();

			Yii::app()->user->setFlash('dashboard',MongoCmsModule::t('Permissions have been recreated.'));
			Yii::app()->end();
		}
	   	else
		  $this->render('permissions_recreate');
	}

	/**
	 * Import data by including the script located in config/mongocms/
	 * If file param is not set, import_data.php will be used
	 *
	 * @return
	 */
	public function actionImportData()
	{
		Yii::app()->mongocmsCheckAccess('importData');

		$cfgFile = $_GET['file'];
		if (empty($cfgFile))
			$cfgFile = 'import_data';

		$config = new MongoCmsConfiguration();
		$config->includeScript($cfgFile);
	}

	/**
	 * Display the backend home
	 */
	public function actionDashboard()
	{
		if (Yii::app()->user->isGuest)
			Page::model()->accessDenied();

		$this->render(Page::model()->getView('dashboard'));
	}

    /**
     * Returns the data model based on the primary key given in the GET variable.
     * If the data model is not found, an HTTP exception will be raised.
     *
     * @param integer $ the ID of the model to be loaded
     */
    public static function loadModel($id)
    {
        $model = Page::model()->findByPk(new MongoID($id));
        if ($model === null)
            throw new CHttpException(404, 'The requested page does not exist.');

        return self::checkModelClass($model);
    }

    /**
     * Converts a Page object to the correct modelclass object
     *
     * @param mixed $model
     * @return
     */
    public static function checkModelClass($model)
    {
        $modelClass = $model->modelclass;
        if ($modelClass != get_class($model)) {
            $result = $model->createInstance($modelClass, $model->attributes, $model->scenario);
            $result->setIsNewRecord(false);
            $result->_id = $model->_id;
            return $result;
        }

        return $model;
    }

    /**
     * Performs the AJAX validation.
     *
     * @param CModel $ the model to be validated
     */
    protected function performAjaxValidation($model)
    {
        if (isset($_POST['ajax']) && $_POST['ajax'] === 'mongopage-form') {
            echo CActiveForm::validate($model);
            Yii::app()->end();
        }
    }

}