<?php

/**
 * MongoCmsUtil.php
 *
 * Static helper functions for rendering, creating links ...
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsUtil
{
    const ID_DOCROUTEFORMELEM = 'id_docroute_frmelem';


    /**
     * Check if an extension file exists
     *
     * @param mixed $extension
     * @return
     */
    public static function extensionExists($extension)
    {
        $ckPath = Yii::getPathOfAlias($extension);
        return file_exists($ckPath . '.php');
    }


	/**
	 * Helper function to explode a text with different separators into an array
	 * Used for search, splitting tags
	 *
	 * @link http://php.net/manual/de/function.preg-split.php
	 * comment from buzoganylaszlo at yahoo dot com
	 *
	 * @see MongoCmsContentProvider::getKeywordContent
	 *
	 * @param string $text
	 * @return array
	 */
	public static function splitWords($text)
	{
		return preg_split("/[\s,]*\\\"([^\\\"]+)\\\"[\s,]*|" . "[\s,]*'([^']+)'[\s,]*|" . "[\s,]+/",
		                $text, 0, PREG_SPLIT_NO_EMPTY | PREG_SPLIT_DELIM_CAPTURE);
	}

    /**
     * Loads the config for a formelement
     * Searchpath
     *
     * @see MongoCmsConfiguration
     * @param mixed $element
     * @return
     */
    public static function getFormelementsConfig($element)
    {
        $config = new MongoCmsConfiguration();
        if (!$config->loadModuleConfig('mongocms', 'formelements'))
            return null;
        else
            return $config[$element];
    }

    /**
     * Get a formatted datetime
     * Set $timeWidth = null for day only
     *
     * @param unix $ timestamp $timestamp
     * @param string $dateWidth
     * @param string $timeWidth
     * @return string
     */
    public static function getFormattedDateTime($timestamp, $dateWidth = 'medium', $timeWidth = 'short')
    {
        return Yii::app()->dateFormatter->formatDateTime($timestamp, $dateWidth, $timeWidth);
    }

    /**
     * Textarea as CKEditor if extension is installed
     *
     * @param CController $controller
     * @param CActiveForm $form
     * @param EMongoSoftDocument $model
     * @param string $attribute
     * @return
     */
    public static function Textarea($controller, $form, $model, $attribute)
    {
        $extension = 'mongocms.extensions.ckeditor.CKEditor';
        if (self::extensionExists($extension)) {
            $config = self::getFormelementsConfig('CKEditor');
            $params = array_merge($config['options'], array('model' => $model, 'attribute' => $attribute));
            $controller->widget($extension, $params, $config['htmlOptions']);
        } else
            echo $form->textArea($model, $attribute, array('rows' => 10, 'cols' => 50));
    }

    /**
     * A Datetime field widget: extension EJuiDateTimePicker
     *
     * @param CController $controller
     * @param CActiveForm $form
     * @param EMongoSoftDocument $model
     * @param string $attribute
     */
    public static function DateTimePicker($controller, $form, $model, $attribute)
    {
        $extension = 'mongocms.extensions.timepicker.timepicker';
        if (self::extensionExists($extension)) {
            $config = self::getFormelementsConfig('timepicker');
            // convert attribute value from unix timestamp to formatted datetime
            $model->$attribute = self::getFormattedDateTime($model->$attribute);
            // EJuiDateTimePicker doen't like en_us
            $language = Yii::app()->language;
            // if (strpos($language,'_') !== false)
            // list($language,$dummy) = explode('_',$language);
            $controller->widget($extension, array(
                    'model' => $model,
                    'name' => $attribute,
                    // 'language' => $language,
                    // 'options' => $config['options'],
                    // 'htmlOptions' => $config['htmlOptions'],
                    ));
        } else
            echo $form->textField($model, $attribute);
    }

    /**
     * The MultiFileUpload widget
     *
     * @param CController $controller
     * @param CActiveForm $form
     * @param EMongoSoftDocument $model
     * @param string $attribute
     */
    public static function MultiFileUpload($controller, $form, $model, $attribute)
    {
        $config = self::getFormelementsConfig('MultiFileUpload');
        $config['name'] = 'new_' . $attribute;
        $config['accept'] = $model->getAllowedFileExtensions();
        $controller->widget('CMultiFileUpload', $config);
    }

    /**
     * Renders the attachments in forms
     *
     * @see views/admin/_form.php
     * @see views/user/_form.php
     *
     * @see MongoCmsFileWidget to list attachments
     *
     * @param CController $controller
     * @param CActiveForm $form
     * @param EMongoSoftDocument $model
     * @param string $attribute
     * @param array $htmlOptions
     */
    public static function listAttachmentsForm($controller, $form, $model, $viewName = '_attachments', $htmlOptions = array())
    {
        $controller->widget('mongocms.components.MongoCmsFileWidget', array(
                'method' => 'renderEditForm',
				'form' => $form,
                'model' => $model,
                'viewName' => $viewName,
                'htmlOptions' => $htmlOptions,
                ));
    }

	/**
	 * Renders the attachments in page view
	 *
	 * @see views/content/_page.php
	 * @see MongoCmsFileWidget to list attachments
	 *
	 * @param CController $controller
	 * @param EMongoSoftDocument $model
	 * @param string $attribute
	 * @param array $htmlOptions
	 */
	public static function listAttachmentsPage($controller, $model,$viewName = '_attachments', $htmlOptions = array())
	{
		$controller->widget('mongocms.components.MongoCmsFileWidget', array(
		        'method' => 'renderView',
		        'model' => $model,
		        'viewName' => $viewName,
		        'htmlOptions' => $htmlOptions,
		        ));
	}

    /**
     * Renders the dropdown list for the status items in _form.php
     *
     * @param CActiveForm $form
     * @param Page $model
     * @param array $htmlOptions
     */
    public static function statusDropDown($form, $model, $htmlOptions = array())
    {
        echo $form->labelEx($model, 'status');
        echo $form->dropDownList($model, 'status', $model->getStatusItems(), $htmlOptions);
        echo '&nbsp;' . $model->getAttributeHint('status');
        echo $form->error($model, 'status');
    }

    /**
     * Renders the dropdown list for the weight in _form.php
     *
     * @param CActiveForm $form
     * @param Page $model
     * @param integer $min
     * @param integer $max
     * @param array $htmlOptions
     */
    public static function weightDropDown($form, $model, $min = - 10, $max = 10, $htmlOptions = array())
    {
        $weight = array();
        for ($i = $min; $i <= $max; $i++)
        $weight[$i] = $i;

        echo $form->labelEx($model, 'weight');
        echo $form->dropDownList($model, 'weight', $weight, $htmlOptions);
        echo $form->error($model, 'weight');
    }

    /**
     * Form elements for handling docrouteid and docroute
     * Using ajax for loading controls on dropdownlist change
     *
     * @see DocRouteController.actionFormElements
     * @param CController $controller
     * @param CActiveForm $form
     * @param Page $model
     * @param array $htmlOptions
     */
    public static function docrouteFormElements($controller, $form, $model, $htmlOptions = array())
    {
        $staticDocRoutes = $model->getStaticDocRoutes();

        if (!empty($staticDocRoutes)) { //render static routes as label (if one) or dropDownList
            echo CHtml::label(MongoCmsModule::t('Route'), 'docroute');
            if (count($staticDocRoutes) == 1)
            {
                $docroute = key($staticDocRoutes);
            	$model->docroute = $docroute; //important: otherwise it will be an array
                echo CHtml::encode($docroute);
                echo $form->hiddenField($model,'docroute');
            }else
                echo $form->dropDownList($model, 'docroute', $staticDocRoutes, $htmlOptions);
        }
        else
        {
            $menuData = Menu::model()->getDocRouteIds();

            $data = CHtml::listData($menuData, 'docrouteid', 'title');

        	$selectedDocRouteId = '';

        	//if is update: find next available menu as selected menu
			if (!empty($model->docroute) && !empty($data))
			{
				$docRoutedata = CHtml::listData($menuData, 'docrouteid', 'docroute');
				$intersectRoute = DocRoute::getIntersectRoute($model->docroute,end($docRoutedata));

				if (!empty($intersectRoute))
					$selectedDocRouteId = array_search($intersectRoute, $docRoutedata);
        	}

            $data = array_merge(array('' => 'none'), $data);

            $url = Yii::app()->controller->createUrl('/mongocms/docroute/formelements',
            	array('modelclass'=>$model->getModelClass(),
                      'id'=>$model->_id,
                      'docrouteid'=>$selectedDocRouteId,
					  ));

            echo CHtml::label(MongoCmsModule::t('Menu'), 'docrouteid');

            $ajaxOptions = array(
                'type' => 'POST',
                'url' => $url,
                'update' => '#' . self::ID_DOCROUTEFORMELEM,
                );

            $htmlOptions = array_merge($htmlOptions, array('ajax' => $ajaxOptions));
            echo CHtml::dropDownList('select_docrouteid', $selectedDocRouteId, $data, $htmlOptions);

        	//div for ajax response
        	$span = CHtml::tag('span',array('id'=>self::ID_DOCROUTEFORMELEM));
        	echo CHtml::tag('div',array('class'=>'row'),$span);

            // load the docroute input for 'none' on show
            // @see DocRouteController.actionFormElements
            Yii::app()->clientScript->registerScript('load_docrouteElem', CHtml::ajax($ajaxOptions), CClientScript::POS_READY);
        }
    }

	public static function docrouteFormElementsCreateMenu($controller, $form, $model, $htmlOptions = array())
	{
		$staticDocRoutes = $model->getStaticDocRoutes();

		if (!empty($staticDocRoutes)) { //render static routes as label (if one) or dropDownList
			echo CHtml::label(MongoCmsModule::t('Route'), 'docroute');
			if (count($staticDocRoutes) == 1) {
				$docroute = key($staticDocRoutes);
				echo CHtml::encode($docroute);
				echo CHtml::hiddenField('docroute', $docroute);
			}else
				echo $form->dropDownList($model, 'docroute', $staticDocRoutes, $htmlOptions);
		}
		else
		{
			$data = array('' => MongoCmsModule::t('New docroute'));
			if (($cursor = $model->getDocRoutesOnly()) !== false && $cursor->count()) {
				$data['existing'] =  MongoCmsModule::t('Existing docroute');
			}


			$url = Yii::app()->controller->createUrl('/mongocms/docroute/formelementscreatemenu');

			echo CHtml::label(MongoCmsModule::t('Select'), 'docroute');

			$ajaxOptions = array(
			    'type' => 'POST',
			    'url' => $url,
			    'update' => '#' . self::ID_DOCROUTEFORMELEM,
			    );

			$htmlOptions = array_merge($htmlOptions, array('ajax' => $ajaxOptions));
			$selected = '';

			echo CHtml::dropDownList('select_existingdocroute', $selected, $data, $htmlOptions);

			//div for ajax response
			$span = CHtml::tag('span',array('id'=>self::ID_DOCROUTEFORMELEM));
			echo CHtml::tag('div',array('class'=>'row'),$span);

			Yii::app()->clientScript->registerScript('load_menudocrouteElem', CHtml::ajax($ajaxOptions), CClientScript::POS_READY);
		}
	}

    /**
     * Print the AddToFavorites or RemoveFromFavorites link
     *
     * @param Page $model
     * @param string $reponseId
     * @return
     */
    public static function userReferencesLink($model,$type,$htmlOptions = array())
    {
    	if (Yii::app()->user->isGuest)
    		return;

		if (($model instanceof Page) && $model->getSettingsValue('showFavoritesLink') && isset($model->_id))
        {
            $user = Yii::app()->mongocmsUser();
			if (isset($user))
			{
				$action = $user->hasReference($model->_id) ? 'remove' : 'add';
				self::renderUserReferencesLink($action,
					                           $model->_id,
					                           $type,
					                           $model->getModelClass(),
					                           $model->title,
					                           $model->modified,
					                           $htmlOptions
					                           );
			}
        }
    }

	/**
	 * Print the reference link
	 * For use in ContentController too
	 *
	 * @param mixed $actionAdd
	 * @param mixed $itemid
	 * @param mixed $type
	 * @param mixed $modelclass
	 * @param string $title
	 */
	public static function renderUserReferencesLink($action,$itemid,$type,$modelclass,$title,$modified,$htmlOptions = array())
	{
		$params = array(
						'action' => $action,
						'itemid' => $itemid,
						'type' => $type,
						'modelclass' => $modelclass,
	                    'title' => $title,
	                    'modified' => $modified,
	                    );

		switch($action){
			case 'add':
				$label = MongoCmsModule::t('Add to favorites');
				break;
			case 'remove':
			case 'removelist':
				$label = MongoCmsModule::t('Remove from favorites');
				break;
			default:
				 throw new CHttpException(400, 'Invalid request.');
		} // switch

		$reponseId = 'refid_'.$itemid;

		$route = Yii::app()->mongocmsControllerRoute('user');
		$url = Yii::app()->createUrl("/$route/references",$params);

//cannot use CHtml::ajaxLink because this registers a script
$onClick =<<<EOP
					$.ajax({
					  url: '$url',
					  cache: false,
					  success: function(response){
					    $('#$reponseId').html(response);
					  }
					});
EOP;

		$htmlOptions = array_merge($htmlOptions,
			array('href'=>'#',
				    'onclick'=>$onClick,
				));

		$content = CHtml::tag('a',$htmlOptions,$label);

		echo CHtml::tag('span',array('id'=>$reponseId,'class'=>'right'),$content);
	}

	/**
	 * Returns a link for an action of a model: create, delete ...
	 * Contenttype access for the action will be checked
	 *
	 * @param Page $model
	 * @param string $action
	 * @param string $label
	 * @param array $urlParams
	 * @param array $htmlOptions
	 * @return link
	 */
	public function getModelActionLink($model,$action,$label = null,$controllerId = null,$subModuleId='',$urlParams = array(),$htmlOptions=array())
	{
		if (($url = $model->getActionItemUrl($action,$controllerId,$subModuleId,$urlParams)) !== false)
		{
			if (!isset($label))
				$label = MongoCmsModule::t($model->getModelClass()) . ' ' . MongoCmsModule::t($action);

			return CHtml::link(MongoCmsModule::t($label),$url,$htmlOptions);
		}

		return '';
	}

	/**
	 * Get a link to a controller action
	 *
	 * @param string $controllerId
	 * @param string $route
	 * @param string $label
	 * @param array $urlParams
	 * @param array $htmlOptions
	 * @return CHtml::link
	 */
	public static function getControllerActionLink($controllerId,$route,$label,$urlParams = array(),$htmlOptions=array())
	{
		$url = Menu::getControllerActionUrl($controllerId,$route,$urlParams);
        return empty($url) ? '' : CHtml::link(MongoCmsModule::t($label),$url,$htmlOptions);
	}

	/**
	 * Returns a link for a CRUD action of a model: create, delete ...
	 * Contenttype access for the action will be checked
	 *
	 * @param Page $model
	 * @param string $action
	 * @param string $label
	 * @param array $urlParams
	 * @param array $htmlOptions
	 * @return link
	 */
	public function getModelCrudActionLink($model,$action,$label = null, $prefix=null, $htmlOptions=array())
	{
        $user = Yii::app()->mongocmsUser();
		$permissions = array($action,$action.'Own',$action.'Any',$action.'SameRole',$action.'SameUsertype');

		if ($model->checkAccess($permissions, array('model' =>$model),true))
		{
			if (!isset($label))
				$label = ucfirst($action);

			$link = self::getModelActionLink($model,$action,
				                     $label,null,'',
				                     array('id'=>$model->_id),$htmlOptions);

			if (!empty($prefix) && !empty($link))
				$link = $prefix . $link;

			return $link;
		}

		return '';
	}

	/**
	 * Returns the link to the action 'tag' of the ContentController
	 *
	 * @param string $tag
	 * @param integer $tagCount
	 * @param array $htmlOptions
	 * @return string
	 */
	public static function getTagLink($tag,$tagCount=0,$htmlOptions=array())
	{
	  if (!empty($tag))
	  {
	  	$urlParams = array('tag'=>$tag);
	  	$url = Menu::getControllerActionUrl('content','tag',$urlParams);

	  	if (!empty($tagCount))
	  	    $label = $tag .' (' . $tagCount . ')';
	  	else
	  		$label = $tag;

	  	return CHtml::link($label,$url,$htmlOptions);
	  }
	}

	/**
	 * Renders the tags of a model
	 *
	 * @param Page $model
	 * @param string $caption
	 * @param array $htmlOptions
	 * @return
	 */
	public static function renderModelTagLinks($model,$caption = null,$htmlOptions=array())
	{
		if (empty($model->tags))
		  return '';

		$tags = is_string($model->tags) ? $model->tagsToArray() : $model->tags;

		if (!empty($caption))
		   echo MongoCmsModule::t($caption) .': ';

		foreach ($tags as $tag)
			echo self::getTagLink($tag,0,$htmlOptions) .' ';
	}


	/**
	 * Create the search form
	 *
	 * @see ContentController.actionSearch
	 * @see views/content/frontpage.php
	 *
	 * @return
	 */
	public static function createSearchForm()
	{
		$model = new SearchForm;
		$config = array(
					    'title' => MongoCmsModule::t('Search content'),

					    'elements' => array(
					        'keywords' => array(
					            'type' => 'text',
					            'maxlength' => 150,
					            'hint' => MongoCmsModule::t('Separate keywords with comma or whitespace'),
					            ),
					        ),

					    'buttons' => array(
					        'search' => array(
					            'type' => 'submit',
					            'label' => MongoCmsModule::t('Search'),
					            ),
					        ),
					    );

		return new CForm($config, $model);
	}

	/**
	 * Renders the searchform partial
	 * @see views/layout/frontpage.php
     *
	 */
	public static function renderSearchForm()
	{
		//create the ContentController
		$controllerArray = Yii::app()->createController('mongocms/content');
		$contentController = $controllerArray[0]; //why an array?
		$form = self::createSearchForm();
		$form->action = $contentController->createUrl('search');
		$contentController->renderPartial('search', array('form'=>$form));
	}

}