<?php

/**
 * MongoCmsFile.php
 *
 * Management of files, images with presets (as attachments to Page)
 * Configuration of presets:
 * @see config/mongocms/imagepresets.php
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsFile extends EMongoGridFS {
	private $_imagePresetConfig;

	public $version;
    public $orgfilename;
    public $type;
    public $metadata;
    public $ownermodelclass; //have to check contenttype permissions on download
    public $ownerpermissions; //a copy of the content permissions of the parent document
    public $imagepresets; //array of _id from preset images

    /**
     * this is similar to the get tableName() method. this returns tha name of the
     * document for this class. this should be in all lowercase.
     */
    public function getCollectionName()
    {
        return 'mongocms_file';
    }

    /**
     * Returns the static model of the specified AR class.
     *
     * @param string $className class name
     * @return CompaniesDb the static model class
     */
    public static function model($className = __CLASS__)
    {
        return parent::model($className);
    }

    /**
     *
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        return array(
            array('filename, orgfilename, title, description, metadata', 'safe'),
            array('permissions,ownermodelclass', 'unsafe'),
            array('filename, orgfilename', 'required'),
            );
    }

    /**
     * Performs access check
     * - Check ownerpermissions (a copy of the content permissions of the owner document)
     * - Check contenttype permissions
     *
     * @return boolean true if access is allowed
     */
    public function checkAccess($checkOnly = false)
    {
        if ($this->checkFileAccess('view', $checkOnly) &&
                Page::model($this->ownermodelclass)->checkContenttypeAccess('view', $checkOnly))
            return true;

        return $checkOnly ? false : Page::model()->accessDenied();
    }

    /**
     * Check access to this loaded instance from mongodb
     * If itemNames is an array, all items have to be allowed
     *
     * @param mixed $itemNames
     * @param array $params
     * @return boolean
     */
    public function checkFileAccess($itemName, $checkOnly = false)
    {
        if (empty($itemName))
            return $checkOnly ? false : Page::model()->accessDenied();
        // if no permissions assign to this content, all operations are allowed
        if (empty($this->ownerpermissions) || empty($this->ownerpermissions['roles']))
            return true;
        // all access allowed for root, even if invalid item
        //  @see MongoCmsUserIdentity->authenticate(), config/default_users.php
        if (Yii::app()->mongocmsIsRootUser())
            return true;

        $userRoles = Yii::app()->mongocmsCurrentUserRoles();

        foreach ($userRoles as $role => $description)
        {
            if (!empty($this->ownerpermissions['roles'][$itemName]) &&
                    in_array($role, $this->ownerpermissions['roles'][$itemName]))
                return true;
        }

        return $checkOnly ? false : Page::model()->accessDenied();
    }

    /**
     * Atomic update of the permissions array
     * If hasImagePresets update all imagepresets too
     *
     * @param array $permissions
     */
    public function updatePermissions($permissions)
    {
        if (!($this->ownerpermissions === $permissions))
        {
            $this->ownerpermissions = $permissions;
            $action = array('$set' => array('ownerpermissions' => $permissions));
            $this->getCollection()->update(array('_id' => new MongoID($this->_id)), $action);

            if ($this->hasImagePresets()) {
                foreach ($this->imagepresets as $preset => $id)
                $this->getCollection()->update(array('_id' => new MongoID($id)), $action);
            }
        }
    }

    /**
     * Generate the download link for a attachment
     *
     * @see FileController
     * @see MongoCmsFileWidget
     * @param string $text
     * @param string $attachment_id
     * @param array $htmlOptions
     * @return string Html link
     */
    public static function downloadLink($text, $fileId, $htmlOptions = array())
    {
        $params = array('fid' => (string)$fileId);
        $route = Yii::app()->mongocmsControllerRoute('file');
        $url = Yii::app()->createUrl($route . '/download', $params);

        $htmlOptions['href'] = $url;
        return CHtml::tag('a', $htmlOptions, $text);
    }

    /**
     * Generate the view url for a attachment
     *
     * @see FileController
     * @see MongoCmsFileWidget
     * @param string $fileId
     * @return string Url
     */
    public static function viewUrl($fileId)
    {
        $params = array('fid' => (string)$fileId);
        $route = Yii::app()->mongocmsControllerRoute('file');
        return Yii::app()->createUrl($route . '/view', $params);
    }

    /**
     * Generate preset images and add the _id to the property imagepresets
     *
     * @return
     */
    public function beforeSave()
    {
        if (parent::beforeSave())
        {
            $this->version = Yii::app()->mongocmsVersion();
			$this->addImagePresets();
            return true;
        }
		else
            return false;
    }

    /**
     * Delete the files from preset images
     */
    protected function beforeDelete()
    {
        if (parent::beforeDelete())
        {
            if (!empty($this->imagepresets))
                foreach ($this->imagepresets as $presetName => $fileid) {
                $presetFile = MongoCmsFile::model()->findByPk(new MongoID($fileid));

                if ($presetFile instanceof MongoCmsFile)
                    $presetFile->delete();
            }
            return true;
        }
		else
            return false;
    }

    /**
     * Set the configuration for creating image presets
     *
     * @param mixed $presets
     * @return
     */
    public function setImagePresetsConfig($imagePresetsConfig)
    {
        $this->_imagePresetConfig = $imagePresetsConfig;
    }

    /**
     * Check if image presets exist
     *
     * @return boolean
     */
    public function hasImagePresets()
    {
        return $this->isImage() && !empty($this->imagepresets);
    }

    /**
     * Create all preset images
     * Assign the images to property imagepresets
     */
    public function addImagePresets()
    {
        if ($this->isImage() && !empty($this->_imagePresetConfig) &&
                file_exists($this->filename))
        {
            $orgimage = Yii::app()->image->load($this->filename);

            foreach ($this->_imagePresetConfig as $presetName => $presetActions)
            {
                $image = clone $orgimage;
                // execute the image methods
                foreach ($presetActions['actions'] as $method => $params)
                call_user_func_array(array($image, $method), $params);

                $presetId = $this->savePresetImage($image, $presetName);

                if (isset($presetId))
                {
                    $existingPresets = is_array($this->imagepresets)
                    ? $this->imagepresets : array();
                    $newPreset = array($presetName => $presetId);
                    $this->setAttributes(array('imagepresets' =>
                            array_merge($existingPresets, $newPreset)),
                        false);
                }
            }
        }
    }

    /**
     * Save the generated preset image to disc
     * load it into GridFs with all parameters
     * return the new _id
     *
     * Workaround for direct stream to GridFs without saving to file???
     */
    protected function savePresetImage(&$image, $presetName)
    {
        $presetFilename = $presetName . '_' . $this->orgfilename;
        $presetFile = dirname($this->filename) . DIRECTORY_SEPARATOR . $presetFilename;

        $image->save($presetFile);

        if (file_exists($presetFile))
        {
            $mongoPreset = new MongoCmsFile();
            // copy attributes ownermodelclass, ownerpermissions to preset for checking access
            $mongoPreset->setAttributes(array('ownermodelclass' => $this->ownermodelclass,
                    'ownerpermissions' => $this->ownerpermissions),
                false);
            if ($mongoPreset->loadFromFile($presetFile, $this->orgfilename, $presetName) && $mongoPreset->save())
            {
                unlink($presetFile); //remove from disc
                return $mongoPreset->_id;
            }
        }
    }

    /**
     * Load from File and set the properties
     */
    public function loadFromFile($filename, $orgFilename = '', $imagePresetName = '')
    {
        if (is_file($filename))
        {
            $this->filename = $filename;
            $this->orgfilename = empty($orgFilename) ? basename($filename) :$orgFilename;
            $this->type = empty($imagePresetName) ? 'attachment' : 'image_' . $imagePresetName;
            $ext = pathinfo($this->orgfilename, PATHINFO_EXTENSION);

            $metadata = array(
                'type' => $this->mimeTypeFromFile($this->orgfilename, $ext),
                'ext' => $ext,
                'size' => filesize($this->filename),
                );

            $this->setAttributes(array('metadata' => $metadata), false);
            return true;
        }
        return false;
    }

    /**
     * Detects the mimetype of a file
     *
     * @param mixed $file
     * @return
     */
    public function mimeTypeFromFile($file, $ext = '')
    {
    	$mimeFilePath =	Yii::getPathOfAlias('system.utils.mimeTypes');
        $mimeType = CFileHelper::getMimeType($file,	$mimeFilePath);
    	return $mimeType;
    }

    /**
     * Check by mimetype if it is an image
     */
    public static function isImageMimeType($mimeType)
    {
        return strpos($mimeType, 'image/') === 0;
    }

    /**
     * Check by mimetype if this file is an image
     */
    public function isImage()
    {
        return self::isImageMimeType($this->metadata['type']);
    }
}