<?php

/**
 * MongoCmsContentProvider.php
 *
 * A collection of usefull data output from the cms
 *
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsContentProvider extends CComponent
{
	public $pagination;


	/**
	 * Get content for the frontpage
	 * Provide all published Pages starting with docroute = $rootDocRoute
	 *
	 * @param string $rootDocRoute
	 * @param array $attributes
	 * @return MongoCmsDataProvider
	 */
	public static function getFrontpageContent($rootDocRoute,$limit=null,$attributes=array())
	{
		$config = (array(
					'modelclass' => 'Content',
					'modelclassOnly' => true,
					'rootDocRoute' => $rootDocRoute,
					'publishedOnly' => true,
					'select' => $attributes,
					'limit' => $limit,
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

		return new MongoCmsDataProvider($config);
	}


	/**
	 * Get content for the dashboard
	 * Provide all published Pages starting with docroute = $rootDocRoute
	 *
	 * @param string $rootDocRoute
	 * @param array $attributes
	 * @return MongoCmsDataProvider
	 */
	public static function getDashboardContent($rootDocRoute,$limit=null,$attributes=array())
	{
		$config = (array(
					'modelclass' => 'DashContent',
					'modelclassOnly' => true,
					'rootDocRoute' => $rootDocRoute,
					'publishedOnly' => true,
					'select' => $attributes,
					'limit' => $limit,
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

		return new MongoCmsDataProvider($config);
	}


	/**
	 * Get content where the current user is the owner
	 *
	 * @return MongoCmsDataProvider
	 */
	public static function getMyContent($attributes=array())
	{
		$user = Yii::app()->mongocmsUser(true);
		$criteria = array(
			    'owner' => $user->_id,
			);

		$config = (array(
					'criteria' => $criteria,
					'select' => $attributes,
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

        return new MongoCmsDataProvider($config);
	}

	/**
	 * Return references of a specific type of the current user
	 * @see References.php
	 *
	 * @param array $attributes the fields to select
	 * @return MongoCmsDataProvider
	 */
	public function getUserReferences($type,$attributes=array())
	{
		$user = Yii::app()->mongocmsUser(true);

		$criteria = array(
			    '_id' => $user->_id, // references from current user
			    'references.type' => $type,
			);

		$config = (array(
					'criteria' => $criteria,
					'modelclass' => 'User',
					'select' => array_merge($attributes,array('references')),
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

		return new MongoCmsDataProvider($config);
	}

	/**
	 * Return the favorites of the current user
	 *
	 * @param array $attributes the fields to select
	 * @return MongoCmsDataProvider
	 */
	public static function getUserFavorites($attributes=array())
	{
		return self::getUserReferences(Reference::TYPE_FAVORITE,$attributes);
	}


	/**
	 * List of the users online
	 *
	 * @param integer $delta seconds intervall to check
	 * @return MongoCmsDataProvider
	 */
	public static function getUsersOnline($delta = 180)
	{
		$criteria = array(
			    'lastOnline' => array('$gt' => time() - $delta),
			);

		$config = (array(
					'criteria' => $criteria,
					'modelclass' => 'User',
					'select' => array('_id','title'),
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

		return new MongoCmsDataProvider($config);
	}


	/**
	 * Return all found tags in the collection mongocms_content
	 * If asLinks, an array of links to ContentController.actionTag will be returned
	 * Results will be cached if menucache is enabled
	 *
	 * @param string $tag
	 * @param integer $limit
	 * @param array $htmlOptions
	 * @return MongoCmsDataProvider
	 */
	public static function getTags($asLinks = true,$limit=null,$htmlOptions=array())
	{

		// check for cached tags
		// menucache is used (will be flush on every saving of a page)
		$cacheId = 'tags'.(int)$asLinks;

		if (Yii::app()->mongocmsModule()->enableMenuCache)
		{
			if (($items = Yii::app()->mongocmsMenuFromCache($cacheId)) !== false)
				return $items;
		}

		$tags = array();
		$criteria = array(
			    //'tags.'.$tag => array('$exists' => true),
			    'tags' => array('$ne' => null),
			);

		$config = array(
					'criteria' => $criteria,
					'publishedOnly' => true,
					'modelclass' => 'Page',
					'select' => array('tags'),
				  );

		if (isset($limit))
		   $config['limit'] = $limit;


		$dataProvider = new MongoCmsDataProvider($config);
		$items = $dataProvider->getData();

		if (!empty($items))
			foreach ($items as $item)
			{
				if (!empty($item->tags))
				{
					$tagsArray = explode(',',$item->tags);
					foreach ($tagsArray as $tag)
					{
						if (!array_key_exists($tag,$tags))
							$tags[$tag] = 1;
						else
							$tags[$tag] = $tags[$tag] + 1;
					}
				}
			}

		//return the result as links array
		if ($asLinks)
		{
			$tagLinks = array();
			foreach (array_keys($tags) as $tag)
				$tagLinks[] = MongoCmsUtil::getTagLink($tag,$tags[$tag],$htmlOptions);

			if (Yii::app()->mongocmsModule()->enableMenuCache)
				Yii::app()->mongocmsMenuToCache($cacheId, $tagLinks);

			return $tagLinks;
		}

		if (Yii::app()->mongocmsModule()->enableMenuCache)
			Yii::app()->mongocmsMenuToCache($cacheId, $tags);

		return $tags;
	}


	/**
	 * Returns published content with the given tag
	 *
	 * @param string $tag
	 * @return MongoCmsDataProvider
	 */
	public static function getTaggedContent($tag)
	{
		$criteria = array(
			    //'tags.'.$tag => array('$exists' => true),
			    'tags' => $tag,
			);

		$config = (array(
					'criteria' => $criteria,
					'publishedOnly' => true,
					'modelclass' => 'Page',
				  ));

		if (!empty($pagination))
			$config['pagination'] = $this->pagination;

		return new MongoCmsDataProvider($config);
	}


	/**
	 * Returns published content containing the keyword
	 *
	 * @param string $keyword
	 * @return MongoCmsDataProvider
	 */
	public static function getKeywordContent($keywords,$attributes=array())
	{
		if (empty($attributes))
			$attributes = array('title','subtitle','body','author');

		$words = MongoCmsUtil::splitWords($keywords);

		$orArray = array();
		if (!empty($words))
		{
			foreach ($words as $word)
			{
				$regexObj = new MongoRegex('/'.$word.'/i');

				foreach ($attributes as $attribute)
					$orArray[] = array($attribute => $regexObj);

				$orArray[] = array('tags' => $regexObj);
			}

			$criteria = array(
					  '$or' =>  $orArray,
		            );

			$config = (array(
					'criteria' => $criteria,
					'publishedOnly' => true,
					'modelclass' => 'Page',
				  ));

			if (!empty($pagination))
				$config['pagination'] = $this->pagination;

			return new MongoCmsDataProvider($config);
		}

		//return a dataprovider object with no data
		return new CArrayDataProvider(array());

	}


}
