<?php

/**
 * MongoCmsBehavior.php
 *
 * Used for calling methods from main module MongoCmsModule
 * Ensures to call the correct module (not a submodule)
 * Local memory caching for some values
 * Simplify some functions: mongocmsMenuFromCache ...
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsBehavior extends CBehavior
{
	/**
	 * Get the main module
	 *
	 * @return MongoCmsModule
	 */
	public function mongocmsModule()
	{
		static $module;

		if (!isset($module))
			$module = Yii::app()->getModule('mongocms');

		return $module;
	}

	/**
	 * Get the version stored with the items in mongoDB
	 */
	public function mongocmsVersion()
	{
		static $version;

		if (!isset($version))
           $version = $this->mongocmsModule()->version;

		return $version;
	}

	/**
	 * Get the route to the mongocms controller
	 * Check if controller is registered in application controllerMap
	 *
	 * @see config/main.php ->controllerMap
	 *
	 * @return string
	 */
	public function mongocmsControllerRoute($controllerId,$addModulePath=true,$subModuleId='')
	{
		static $path;

		if (!isset($path))
			$path = array();

		$cacheId = $controllerId.$subModuleId.(int)$addModulePath;

		if (!isset($path[$cacheId]))
		{
			$controllerClass = ucfirst($controllerId) . 'Controller';
			$result = $this->mongocmsModule()->getControllerMapKey($controllerClass,!empty($subModuleId));

			if (empty($result)) //controller is not mapped
			{
				$subModuleId = strtolower($subModuleId);

				$controllerPath = empty($subModuleId) ? 'application.modules.mongocms.controllers'
					                                  : "application.modules.mongocms.modules.$subModuleId.controllers";

				$basePath = Yii::getPathofAlias($controllerPath);
				$classFile =  $basePath.DIRECTORY_SEPARATOR.$controllerClass.'.php';
				$mongocmsRoute = $addModulePath ? 'mongocms/' : '';

				if (!empty($subModuleId))
					$subModuleId = $subModuleId .'/';

				$result = is_file($classFile) ? $mongocmsRoute.$subModuleId.$controllerId : '';
			}

			$path[$cacheId] = $result;
		}

		return 	$path[$cacheId];
	}

	/**
	 * Get the configured mongodb
	 */
	public function mongocmsDb()
	{
		static $mongoDb;

		$connId = $this->mongocmsModule()->mongoConnectionId;
		if ($mongoDb === null)
			$mongoDb = Yii::app()->getComponent($connId);

		return $mongoDb->getDbInstance();
	}

	/**
	 * Get a collection from the configured mongodb
	 *
	 * @param string $name
	 */
	public function mongocmsCollection($name)
	{
		static $mongoCollections;

		if (!isset($mongoCollections[$name]))
			$mongoCollections[$name] = $this->mongocmsDb()->selectCollection($name);

		return $mongoCollections[$name];
	}


	/**
	 * The global Authmanager of mongocms, independent of contenttypes
	 *
	 * @param boolean $loadFromDB
	 * @return MongoCmsAuthmanager
	 */
	public function mongocmsAuthManager($loadFromDB = true)
	{
		static $authManager;

		if (!isset($authManager))
			$authManager = array();

		if (!isset($authManager[(int)$loadFromDB]))
			$authManager[(int)$loadFromDB] = $this->mongocmsModule()->getAuthManager($loadFromDB);

		return $authManager[(int)$loadFromDB];
	}

	/**
	 * Check if is root user
	 *
	 * @return boolean
	 */
	public function mongocmsIsRootUser()
	{
		return !Yii::app()->user->isGuest &&
			   isset(Yii::app()->user->isRoot) &&
			   Yii::app()->user->isRoot;
	}

	/**
	 * Get the model of the current user
	 *
	 * @param mixed $id
	 * @return
	 */
	public function mongocmsUser($abortIfNotExists = false)
	{
		static $user; //cache user model

		if (Yii::app()->user->isGuest)
		{
			if ($abortIfNotExists)
				Page::accessDenied();
			else
				return null;
		}

		if (!isset($user))
		{
			$userid = Yii::app()->user->id;
			$user = User::model()->findByPk(new MongoID($userid));
			if (!empty($user))
			  $user->updateLastOnline();
		}

		if (empty($user) && $abortIfNotExists)
			throw new CHttpException(404, 'Not Found.');

		return $user;
	}

	/**
	 * Get the defined roles from main module.
	 * If $includeAuthRoles, the roles guest and authenticated will be added
	 *
	 * @param boolean $includeAuthRoles
	 * @return array
	 */
	public function mongocmsDefinedRoles($includeAuthRoles=true,$fromCache = true)
	{
		static $roles;

		if (!$fromCache)
			unset($roles);

		if (!isset($roles))
			$roles = $this->mongocmsModule()->getDefinedRoles($includeAuthRoles);

		return $roles;
	}

	/**
	 * Get the roles of the current user as value pairs ($role => $description)
	 * Note: mongocmsUser()->roles does only return the $roles
	 *
	 * @return array ($role=>$description, ...)
	 */
	public function mongocmsCurrentUserRoles()
	{
		static $roles;

		if (!isset($roles))
			$roles = Yii::app()->user->isGuest || !isset(Yii::app()->user->roles)
				     ? self::getRoleTypeGuest() : Yii::app()->user->roles;

		return $roles;
	}

	/**
	 * The definition of the guest role
	 *
	 * @return array
	 */
	public static function getRoleTypeGuest()
	{
		return array('guest'=>'Guest');
	}

	/**
	 * The definition of the authenticated role
	 *
	 * @return
	 */
	public static function getRoleTypeAuthenticated()
	{
		return array('authenticated'=>'Authenticated');
	}


	/**
	 * Check access to global defined permissions in mongocms
	 * (managePermissions ...)
	 *
	 * @param mixed $itemNames
	 * @param array $params
	 * @param mixed $checkOnly
	 * @param mixed $or
	 * @return
	 */
	public static function mongocmsCheckAccess($itemNames, $params = array(), $checkOnly = false, $or = true)
	{
		return self::mongocmsModule()->checkAccess($itemNames, $params, $checkOnly, $or);
	}


	/**
	 * Generate the id for menu caching depending on the current user roles
	 *
	 * @return string
	 */
	protected static function generateRolesCacheId($cacheId)
	{
	   return $cacheId . md5(serialize(self::mongocmsCurrentUserRoles()));
	}

	/**
	 * Load the menu items for a specific menu from cache
	 *
	 * mongocmsMenuCache is created in MongoCmsModule.init
	 *
	 * @see mongocms/config/mongocms/createmenuitems_main.php
	 *
	 * @param string $menuId
	 * @return array or false
	 */
	public static function mongocmsMenuFromCache($menuId)
	{
		$cacheId = self::generateRolesCacheId($menuId);
		$value=Yii::app()->mongocmsMenuCache->get($cacheId);
		return $value !== false ? unserialize($value) : false;
	}

	/**
	 * Save the menu items for a specific menu to cache
	 * A own cacheid is used for current user roles
	 *
	 * mongocmsMenuCache is created in MongoCmsModule.init
	 *
	 * @see mongocms/config/mongocms/createmenuitems_main.php
	 *
	 * @param string $menuId
	 */
	public static function mongocmsMenuToCache($menuId,$items)
	{
		$cacheId = self::generateRolesCacheId($menuId);
		Yii::app()->mongocmsMenuCache->set($cacheId, serialize($items));
	}

	/**
	 * Load a specific page from cache
	 * mongocmsPageCache is created in MongoCmsModule.init
	 *
	 * @param string $pageId
	 * @return array or false
	 */
	public static function mongocmsPageFromCache($pageId)
	{
		$cacheId = self::generateRolesCacheId($pageId);
		$value=Yii::app()->mongocmsPageCache->get($cacheId);
		return $value !== false ? $value : false;
	}

	/**
	 * Save a specific page to cache
	 * A own cacheid is used for current user roles
	 *
	 * mongocmsPageCache is created in MongoCmsModule.init
	 *
	 * @param string $pageId
	 * @param string $value
	 */
	public static function mongocmsPageToCache($pageId,$value)
	{
		$cacheId = self::generateRolesCacheId($pageId);
		Yii::app()->mongocmsPageCache->set($cacheId, $value);
	}

	/**
	 * Get the defined contenttypes from main- and submodules
	 *
	 * @return array
	 */
	public function mongocmsContentTypes()
	{
		static $contentTypes;

		if (!isset($contentTypes))
			$contentTypes = $this->mongocmsModule()->getContentTypes();

		return $contentTypes;
	}

	/**
	 * Get cached docroute for memory cached access control
	 * @see DocRoute.findAllowedDocumentIds
	 *
	 * @return DocRoute
	 */
	public function mongocmsDocRoute()
	{
		static $docRoute;

		if (!isset($docRoute))
			$docRoute = new DocRoute;

		return $docRoute;
	}


	/**
	 * Return the module where the contenttype is defined
	 *
	 * @return object
	 */
	public function mongocmsModuleFromContenttype($modelClass)
	{
		static $module;

		if (!isset($module))
			$module = array();

		if (!isset($module[$modelClass]))
			$module[$modelClass] = $this->mongocmsModule()->getModuleFromContenttype($modelClass);

		return $module[$modelClass];
	}

	/**
	 * Return the menuItems from config/mongocms/menuitems_MENUID
	 *
	 * @param string $menuId
	 * @return array
	 */
	public function mongocmsCreateMenuItems($menuId)
	{
		static $menus;

		if (!isset($menus))
			$menus = array();

		if (!isset($menus[$menuId]))
			$menus[$menuId] = $this->mongocmsModule()->createMenuItems($menuId);

		return $menus[$menuId];
	}


	/**
	 * Get the viewPath from main module
	 *
	 * @return string
	 */
	public function mongocmsViewPath()
	{
		$this->mongocmsModule()->viewPath;
	}

	/**
	 * Get the layoutpath from main module
	 *
	 * @return string
	 */
	public function mongocmsLayoutPath()
	{
		$this->mongocmsModule()->layoutPath;
	}

	/**
	 * Call main modules install method
	 *
	 * @return
	 */
	public function mongocmsInstall()
	{
		$this->mongocmsModule()->install();
	}


	public function mongocmsUIThemeUrl()
	{
	  return $this->mongocmsModule()->getMongoCmsAssetsUrl() .'/themes';
	}

	public function mongocmsUITheme()
	{
		return $this->mongocmsModule()->uiTheme;
	}

}
