<?php

/**
 * MongoCmsAuthManager.php
 *
 * A variant of my published extension CMongoDbAuthManager
 *
 * A quick hack that extends CPhpAuthManager
 * and overrides loadFromFile, saveToFile
 * The file on disk is replaced by a collection in a mongodb
 *
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsAuthManager extends CPhpAuthManager {
    public $mongoConnectionId = 'mongodb';
    // The authFile property is the collection name
    public $authFile = 'mongocms_permissions';

    private $_authId;
    private $_id; //the MongoId
	private $_version;

    protected static $_emongoDb;
    protected static $_mongocollections = array();

    /**
     * MongoCmsAuthManager::__construct()
     *
     * @param mixed $authId
     */
    public function __construct($authId)
    {
        $this->_authId = $authId;
    }

    /**
     * Get raw MongoDB instance
     *
     * @return MongoDB
     */
    public function getDb()
    {
        if (self::$_emongoDb === null)
            self::$_emongoDb = Yii::app()->getComponent($this->mongoConnectionId);
        return self::$_emongoDb->getDbInstance();
    }

    /**
     * Returns current MongoCollection object
     * By default this method use {@see authFile}
     *
     * @return MongoCollection
     */
    public function getCollection($name = null)
    {
        if (!isset($name)) {
            $name = $this->authFile;
        }

        if (!isset(self::$_mongocollections[$name]))
            self::$_mongocollections[$name] = $this->getDb()->selectCollection($name);

        return self::$_mongocollections[$name];
    }

    /**
     * Loads the authorization data from mongo db
     *
     * @param string $file is the collection name
     * @return array the authorization data
     * @see saveToFile
     */
    protected function loadFromFile($file)
    {
        $collection = $this->getCollection($file);
        $criteria = array('authId' => $this->_authId);
        $result = $collection->findOne($criteria);

        if (empty($result))
            return array();
        // remove _id, because it's not an AuthItem
        if (isset($result['_id']))
        {
            $this->_id = $result['_id'];
            unset($result['_id']);
        }
        // remove authId, because it's not an AuthItem
        if (isset($result['authId']))
        {
            $this->_authId = $result['authId'];
            unset($result['authId']);
        }

    	// remove version, because it's not an AuthItem
    	if (isset($result['version']))
    	{
    		$this->_version = $result['version'];
    		unset($result['version']);
    	}

        return $result;
    }

    /**
     * Saves the authorization data from the collection 'file'
     *
     * @param array $data the authorization data
     * @param string $file the collection name
     * @see loadFromFile
     */
    protected function saveToFile($data, $file)
    {
        $collection = $this->getCollection($file);
        // have to set the _id for scenario update
        if (isset($this->_id))
            $data['_id'] = new MongoId($this->_id);

        $data['authId'] = $this->getAuthId();

    	$data['version'] = Yii::app()->mongocmsVersion();

        $collection->save($data);
        $this->_id = $data['_id'];
    }

    /**
     * Delete an Authmanager record from collection
     *
     * @param mixed $authId
     * @return
     */
    public function delete($authId = null)
    {
        if (!isset($authId))
            $authId = $this->getAuthId();

        if (!empty($authId)) {
            $collection = $this->getCollection();
            $criteria = array('authId' => $authId);
            $collection->remove($criteria);
        }
    }

    /**
     * Check if modelClass permissions are stored in mongodb
     *
     * @param mixed $authId
     * @return
     */
    public function contentTypeExists($modelClass = null)
    {
        if (!isset($modelClass))
            $modelClass = $this->getAuthId();

        $collection = $this->getCollection();
        $criteria = array('authId' => $modelClass);
        return $collection->count($criteria) > 0;
    }

    /**
     * Check if modelClass permissions are stored in mongodb
     *
     * @param mixed $authId
     * @return
     */
    public function instanceExists($authId = null)
    {
        if (!isset($authId))
            $authId = $this->getAuthId();

        $collection = $this->getCollection();
        $criteria = array('authId' => $authId);
        return $collection->count($criteria) > 0;
    }

    /**
     * Set the authId
     *
     * @param string $authId
     */
    public function setAuthId($authId)
    {
        $this->_authId = $authId;
    }

    /**
     * Get the authId
     *
     * @return string
     */
    public function getAuthId()
    {
        return empty($this->_authId) ? 'default' : $this->_authId;
    }

    /**
     * Get the authId
     *
     * @return string
     */
    public function getId()
    {
        return $this->_id;
    }

	/**
	 * Get the version
	 *
	 * @return string
	 */
	public function getVersion()
	{
		return $this->_version;
	}

    /**
     * Check if is initialized with data
     *
     * @return boolean
     */
    public function hasData()
    {
        $model = $this->getCollection()->findOne();
        return !empty($model);
    }

    /**
     * Ensure the operation is not registered
     *
     * @param string $name
     * @param string $description
     * @param string $bizRule
     * @param mixed $data
     * @return
     */
    public function createOperation($name, $description = '', $bizRule = null, $data = null)
    {
        $item = $this->getAuthItem($name);
        return empty($item) ? parent:: createOperation($name, $description, $bizRule, $data) : $item;
    }

    /**
     * Ensure the role is not registered
     *
     * @param string $name
     * @param string $description
     * @param string $bizRule
     * @param mixed $data
     * @return
     */
    public function createRole($name, $description = '', $bizRule = null, $data = null)
    {
        $item = $this->getAuthItem($name);
        return empty($item) ? parent:: createRole($name, $description, $bizRule, $data) : $item;
    }

    /**
     * Ensure the task is not registered
     *
     * @param string $name
     * @param string $description
     * @param string $bizRule
     * @param mixed $data
     * @return
     */
    public function createTask($name, $description = '', $bizRule = null, $data = null)
    {
        $item = $this->getAuthItem($name);
        return empty($item) ? parent:: createRole($name, $description, $bizRule, $data) : $item;
    }


	/**
	 * Create a role and add operations as child
	 * The role have to be defined
	 * @see config/mongocms/roleoperations_page.php
	 *
	 * @param string $role
	 * @param array $operations
	 * @param string $bizRule
	 * @param string $data
	 */
	public function addRoleOperations($role,$operations = array(), $bizRule = null, $data = null)
	{
		$definedRoles = Yii::app()->mongocmsDefinedRoles();

		if (array_key_exists($role, $definedRoles))
		{
			$description = $definedRoles[$role];
			$authItem = $this->createRole($role,$description, $bizRule, $data);

			if (!empty($operations))
				foreach ($operations as $operation)
					$authItem->addChild($operation);
		}
	}

	/**
	 * Create the role and add all operations as child
	 * The role have to be defined
	 * Used for admin role
	 * @see config/mongocms/roleoperations_page.php
	 *
	 * @param string $role
	 */
	public function addRoleOperationsAll($role)
	{
		$definedRoles = Yii::app()->mongocmsDefinedRoles();

		if (array_key_exists($role, $definedRoles))
		{
			$description = $definedRoles[$role];
			$authItem = $this->createRole($role,$description, $bizRule, $data);

			foreach ($this->operations as $operation => $obj)
				$authItem->addChild($operation);
		}
	}


	/**
	 * Create the guest role and add operations as child
	 * Used for admin role
	 *
	 * @param string $role
	 */
	public function addGuestRoleOperations($operations)
	{
		$bizRule = 'return Yii::app()->user->isGuest;';
		$role = key(MongoCmsBehavior::getRoleTypeGuest());
		$this->addRoleOperations($role,$operations, $bizRule);
	}


	/**
	 * Create the authenticated role and add operations as child
	 * Used for admin role
	 *
	 * @param string $role
	 */
	public function addAuthenticatedRoleOperations($operations)
	{
		$bizRule = 'return !Yii::app()->user->isGuest;';
		$role = key(MongoCmsBehavior::getRoleTypeAuthenticated());
		$this->addRoleOperations($role,$operations, $bizRule);
	}

    /**
     * Remove all operations from the roles
     *
     * @return
     */
    public function removeRoleOperations()
    {
        $rolesObj = $this->getRoles();
        $operationsObj = $this->getOperations();
        // remove all permissions from roles
        foreach($rolesObj as $roleName => $roleAuthItem)
	        foreach($operationsObj as $operationName => $operationAuthItem)
	            $this->removeItemChild($roleName, $operationName);
    }

	/**
	 * Check access to a single item
	 *
	 * @param string $itemName
	 * @param array $params
	 * @return
	 */
	public function _checkUserAccess($itemName,$params=array())
	{
		$authItem = $this->getAuthItem($itemName);
		$roles = Yii::app()->mongocmsCurrentUserRoles();

		if (empty($authItem))
			return false;

		$b = $authItem->getBizRule();
		if ($this->executeBizRule($authItem->getBizRule(), $params, $authItem->getData())) {
			if (in_array($itemName, $this->defaultRoles))
				return true;

			foreach ($roles as $role => $description) {
				$roleObj = $this->getAuthItem($role);
				if (!empty($roleObj) && $roleObj->hasChild($itemName))
					return true;
			}
		}

		return false;
	}


	/**
	 * Check access to operations for the current user
	 * This authmanager for mongocms doesn't work with assignments
	 *
	 * @param string $itemName
	 * @param array $params
	 * @return
	 */
	public function checkUserAccess($itemNames, $params = array(), $checkOnly = false, $or = true)
	{
		if (Yii::app()->mongocmsIsRootUser())
			return true;

		if (empty($itemNames))
			return $checkOnly ? false : Page::accessDenied();

		if (is_string($itemNames))
		{
			$allowed = $this->_checkUserAccess($itemNames, $params);

			Yii::trace('Checked contenttype access "' . $itemNames .
				        '" allowed: ' .$allowed .
				        ' authManager: ' . $this->authId,
				         'AuthManager.checkUserAccess');

			return $allowed ? true : ($checkOnly ? false : Page::accessDenied());
		}
		else
		{
			for($i = 0; $i < count($itemNames); $i++)
			{
				$allowed = $this->_checkUserAccess($itemNames[$i], $params);

				Yii::trace('Checked contenttype access "' . $itemNames[$i] .
					        '" allowed: ' .$allowed .
		                   ' authManager: ' . $this->authId,
		                   'AuthManager.checkUserAccess');

				if (!$or && !$allowed) //AND: all have to be allowed
					return $checkOnly ? false : Page::accessDenied();

				if ($allowed && $or)
					return true;
			}

			return $or ? ($checkOnly ? false : Page::accessDenied()) : true;
		}

		return $checkOnly ? false : Page::accessDenied();
	}


	/**
	 * Is the creator of a content the same usertype as the current user?
	 * Checks if the docroute of the user who created the content
	 * is the same as the docroute of current user
	 *
	 * @return string
	 */
	public static function bizRuleSameUsertype()
	{
		return 'return Yii::app()->user->isGuest ? false : Yii::app()->user->docroute==$params["model"]->ownerdocroute;';
	}

	/**
	 * Share the creator of a content at least one role with the current user?
	 * Note: Yii::app()->user->roles returns the roles as assoziative array
	 *       ($role=>$description ...)
	 *
	 * @return string
	 */
	public static function bizRuleSameUserRole()
	{
		return 'return Yii::app()->user->isGuest ? false : count(array_intersect(array_keys(Yii::app()->user->roles),$params["model"]->ownerroles))>0;';
	}

	/**
	 * Is the owner of a content the same as the current user?
	 *
	 * @return string
	 */
	public static function bizRuleSameOwner()
	{
		return 'return Yii::app()->user->isGuest ? false : Yii::app()->user->id==$params["model"]->owner;';
	}
}

