<?php

/**
 * MongoCmsModule.php
 *
 * The main module of the mongocms
 * Implements functionality for handling the submodules, install, caching ...
 *
 * Expose the public properties to the applications config/main.php if necessary
 *
 *
 * PHP version 5.2+
 *
 * @author Joe Blocher <yii@myticket.at>
 * @copyright 2011 myticket it-solutions gmbh
 * @license New BSD License
 * @category User Interface
 * @package modules.mongocms.MongoCmsModule
 * @version 0.1
 * @since 0.1
 */
class MongoCmsModule extends CWebModule {

	const MENUCACHE_COLLECTIONNAME = 'mongocms_cache_menu';
	const PAGECACHE_COLLECTIONNAME = 'mongocms_cache_page';
	const MONGOCMS_AUTHID = 'mongocms_global'; //authId for the global cms authmanager

	/**
	 * The allowed file extensions for attachments
	 * @var array $allowedFileExtensions
	 */
	public $allowedFileExtensions = array('png','jpg','jpeg','gif','txt','doc','docx','xls','pdf');

	/**
	 * @var boolean $ensureIndexes
	 * ensureIndexes should be set to false in production enviroment
	 * @see Page->init()
	 * @link http://canni.github.com/YiiMongoDbSuite/xhtml/advanced.indexing.html
	 */
	public $ensureIndexes = true;

	/**
	 * The default theme of mongocms
	 * The theming follows the documentation of Yii theming
	 *
	 * Customizing the mongocms
	 *
	 * You can change the value of the theme property (example 'mycms')
	 * Before change to another value you have to copy and rename the folders for themeing, config, assets
	 *
	 *  1. Copy the folder protected/modules/mongocms/themes/mongocms
	 *     to: protected/modules/mongocms/themes/mycms
	 *
	 *  2. Copy the configuration files from protected/modules/config/mongocms
	 *     to: protected/modules/config/mycms
	 *     Note: You have to do the same for the configuration files of the submodules too: slideshow ...
	 *
	 *  3. Copy the folder assets/mongocms to assets/mongocms to assets/mycms
	 *
	 *  4. Set the value of the theme property below to 'mycms'
	 *
	 *  Now you can alter all views and configurations for your need,
	 *  add custom ui themes or alter the jqueryslidemenu css
	 *
	 * @link http://www.yiiframework.com/doc/guide/1.1/en/topics.theming
	 *
	 * @var string $theme
	 * @since 0.1
	 */
	public $theme = 'mongocms';

	/**
	 * This is the name of the ui theme use with jquery ui components
	 *
	 * mongocms publishes a couple of ui-themes
	 * @see assets/themes
	 *
	 * @var string $uiTheme
	 * @since 0.1
	 */
	public $uiTheme = 'cupertino';

	/**
	 * The location of the theme directory
	 * The structure follows the documentation of Yii theming
	 *
	 * @link http://www.yiiframework.com/doc/guide/1.1/en/topics.theming
	 * @var string $themeBasePath the location of the folder in module mongocms
	 * @since 0.1
	 */
	public $themeBasePath = 'mongocms.themes';

	/**
	 * Count of characters when displaying pages in a list
	 * If the body len of the page is greater than $teaserSize
	 * the 'Read more' link will be added
	 *
	 * @var integer $teaserSize
	 * @since 0.1
	 */
	public $teaserSize = 500;

	/**
	 * Configuration for the integrated extension 'image'
	 * @see http://www.yiiframework.com/extension/image
	 *
	 * @var string $imageDriver
	 * @since 0.1
	 */
	public $imageDriver = 'GD'; // GD or ImageMagick
	public $imageMagickDir; // 'pathTo/ImageMagick-x' if driver is ImageMagick

	/**
	 * @var string $mongoConnectionId the configured application component of the YiiMongoDbSuite
	 * @since 0.1
	 */
	public $mongoConnectionId = 'mongodb';


	/**
	 * If true, the default installation will run on showing the loginform
	 * The installation methods will only be executed if the collections are empty or not exists.
	 * Set $autoInstall to false after first login to increase performance when displaying the loginForm.
	 *
	 * @see LoginForm->init()
	 * @var boolean $autoInstall
	 * @since 0.1
	 */
	public $autoInstall = true;

	/**
	 * Disable menu cache for testing/development,
	 * enable in production enviroment
	 *
	 * @see config/mongocms/createmenuitems_main.php
	 * @var boolean $autoInstall
	 * @since 0.1
	 */
	public $enableMenuCache = false;


	public $defaultController = 'content';
    protected static $_translationSourceId = 'mongocmsMessages';
    protected static $_authManager = array();

	/**
	 * The mongocms version
	 * This intvalue is stored with every mongocms item in the mongodb
	 * This is for preparing upgrades in upcoming releases
	 * Don't change this value
	 *
	 * @var integer $_version
	 * @since 0.1
	 */
	private $_version=1;

	/**
	 * The assets url for global mongocms assets
	 *
	 * @var integer $_version
	 * @since 0.1
	 */
    private $_assetsUrl;

    /**
     * Initialize import and translationSource
     */
    public function init()
    {
		parent::init();

    	//init theme directory
    	Yii::app()->themeManager->basePath = Yii::getPathOfAlias($this->themeBasePath);
    	Yii::app()->theme = $this->theme;

    	//always use the cms layout, for submodules too
    	$this->layoutPath = Yii::getPathOfAlias($this->themeBasePath .'.'.
    		                                    $this->theme .
    		                                    '.views.mongocms.layouts');

    	$this->viewPath = Yii::getPathOfAlias($this->themeBasePath .'.'.
    		$this->theme .
    		'.views.mongocms');

        // should work for submodules too
        $path = $this->isSubModule() ? '.modules.' : '.';
        $path = str_replace('/', $path, $this->id);

        $this->setImport(array($path . '.models.*',
                               $path . '.components.*',
                ));

    	//import the models of the submodules
    	$this->importSubmoduleModels();

        // register MessageSource for translation
    	$components = array(
		    	self::$_translationSourceId => array(
		    	      'class' => 'CPhpMessageSource',
		    	      'basePath' => $this->basePath . DIRECTORY_SEPARATOR . 'messages',
		    	  ),
	     );


    	$components['image'] =  array(
             'class' => 'mongocms.extensions.image.CImageComponent',
            // GD or ImageMagick
             'driver' => $this->imageDriver,
            // ImageMagick setup path
             'params'=>array('directory'=>$this->imageMagickDir),
            );


    	//install CMongoDbCache if no other cache is installed
    	if (!Yii::app()->hasComponent('cache'))
    		$components['cache'] = array('class' => 'mongocms.components.EMongoDBCache');

    	 //install extrac caches for mongocms, allows to full flush without flushing yii cache

		 //install mongocmsMenuCache
    	 $components['mongocmsMenuCache'] = array('class' => 'mongocms.components.EMongoDBCache',
    	                                     	  'collectionName' => self::MENUCACHE_COLLECTIONNAME);

    	//install mongocmsPageCache
    	$components['mongocmsPageCache'] = array('class' => 'mongocms.components.EMongoDBCache',
    	                                     	  'collectionName' => self::PAGECACHE_COLLECTIONNAME);

        Yii::app()->setComponents($components);
    }


	/**
	 * Return the version stored with every item in the mongodb
	 *
	 * @return integer
	 */
	public function getVersion()
	{
		return $this->_version;
	}

	/**
	 * LoginUrl readonly
	 *
	 * @return string
	 */
	public function getLoginUrl()
	{
		$route = Yii::app()->mongocmsControllerRoute('user');
		return Yii::app()->createUrl($route.'/login');
	}

	/**
	 * LogoutUrl readonly
	 *
	 * @return
	 */
	public function getLogoutUrl()
	{
		$route = Yii::app()->mongocmsControllerRoute('user');
		return Yii::app()->createUrl($route.'/logout');
	}

    /**
     * Get the locale date format for validation rules
     */
    public function getValidationRuleDateFormat()
    {
        return Yii::App()->locale->getDateFormat('short');
    }

    /**
     * Get the locale datetime format for validation rules
     */
    public function getValidationRuleDateTimeFormat()
    {
        return $this->getValidationRuleDateFormat() . ' ' .
        Yii::App()->locale->getTimeFormat('short');
    }

    /**
     * Returns the key of requested class in applications controllerMap
     * This is the value of the docroute to request for files or content ...
     * Need if controllerMap key is not set to default
     *
     * @param string $controllerClass : FileController or ContentController
     * @return string
     */
    public function getControllerMapKey($controllerClass,$fromSubmodule = false)
    {
        $map = Yii::app()->controllerMap;

    	if ($fromSubmodule && $this->isSubModule())
    	{
    		$path = str_replace('/', '.modules.', $this->id);
    		$path = 'application.modules.' . $path . '.controllers.' . $controllerClass;
    	}
    	else
    		$path = 'application.modules.mongocms.controllers.' . $controllerClass;

        return array_search($path, $map);
    }


    /**
     * Check if this is a submodule
     *
     * @return boolean
     */
    public function isSubModule()
    {
        return isset($this->parentModule);
    }

    /**
     *
     * @return string the base URL that contains all published asset files of this module.
     */
    public function getMongoCmsAssetsUrl()
    {
    	if ($this->_assetsUrl === null)
    		$this->_assetsUrl = Yii::app()->getAssetManager()->publish(Yii::getPathOfAlias('mongocms.assets.'.$this->theme));

        return $this->_assetsUrl;
    }


	/**
	 * Build the items of a menu from a config file createmenuitems_MENUID.php
	 *
	 * @see config/mongocms/createmenuitems_main.php
	 *
	 * @param string $menuId, the id of a menu
	 * @return array
	 */
	public function createMenuItems($menuId)
	{
		$config = new MongoCmsConfiguration();
		if ($config->loadModuleConfig($this,'createmenuitems_' . $menuId))
		  	return $config->toArray();
		else
			return array();
	}

	/**
	 * Helper function to merge an array with methodresults from submodules
	 *
	 * @param mixed $method
	 * @param mixed $mergeArray
	 * @return
	 */
	public function mergeSubmodulesArray($method,&$mergeArray)
    {
		if (!$this->isSubmodule())
		{
			foreach ($this->modules as $id => $array)
			{
				$subModule = $this->getModule($id);
				if (($subModule instanceof MongoCmsModule) &&
					method_exists($subModule, $method))
					$result = call_user_func(array($subModule, $method));
				$mergeArray = array_merge($mergeArray, $result);
			}
		}
	}

	/**
	 * Returns all registered contenttypes
	 * for all submodules too
	 *
	 * @return array Classname => Label
	 */
	public function getContentTypes($askSubmodules=true,$includeDocRoute = false)
	{
		$result = array('Page'=>'Page',
		                'Menu'=>'Menu',
		                'User'=>'User',
		                'Role'=>'Role',
		                'Content'=>'Frontpage content',
		                'DashContent'=>'Dashboard content',
		               );

		//Expose contenttype docroute only for route (for develment/testing)
		//DocRoute should work in background properly
		if ($includeDocRoute || Yii::app()->mongocmsIsRootUser())
			$result['DocRoute']='DocRoute';

		// collect contentTypes from submodules too
		if ($askSubmodules)
			$this->mergeSubmodulesArray('getContentTypes',$result);

		return $result;
	}


	/**
	 * Return the module where the contenttype is defined
	 *
	 * @return object
	 */
	public function getModuleFromContenttype($modelClass)
	{
		$mainContentTypes = $this->getContentTypes(false,true);
		if (array_key_exists($modelClass,$mainContentTypes))
			return $this;

		if (!$this->isSubmodule())
		{
			$method = 'getContentTypes';
			foreach ($this->modules as $id => $array)
			{
				$subModule = $this->getModule($id);
				if (($subModule instanceof MongoCmsModule) &&
				method_exists($subModule, $method))
				{
					$result = call_user_func(array($subModule, $method));
					if (!empty($result) && array_key_exists($modelClass,$result))
						return $subModule;
				}
			}
		}

	    return null;
	}


	/**
	 * Collect all defined roles from all authmanagers (contenttype, cms)
	 *
	 * @return array
	 */
	public function getDefinedRoles($includeAuthenticatedRoles=true)
	{
		$roles = Role::model()->getDefinedRoles();

		$authenticatedRoles = array_merge(MongoCmsBehavior::getRoleTypeGuest(),
			                              MongoCmsBehavior::getRoleTypeAuthenticated());

		return $includeAuthenticatedRoles ? array_merge($authenticatedRoles,$roles)
			                              : array_diff($roles,$authenticatedRoles);
	}


	/**
	 * Recreate all contenttype permissions from config files 'roleoperations_MODELCLASS'
	 * This method should be called if one of the config files has been changed
	 * @see AdminController.actionRecreatePermissions
	 *
	 */
	public function recreatePermissions()
	{
		if ($this->isSubmodule())
			throw new CExeptions('This method must not be called from submodule');

		foreach ($this->getContentTypes(true,true) as $modelClass => $label)
			Page::model($modelClass)->installContentType(true);

	    $this->installCMSPermissions(true);
	}

	/**
	 * Called when need to import all models from submodules
	 * @see MongoCmsModule.installMongoDb, AdminController->actionPermissions
	 */
	public function importSubmoduleModels()
	{
		if (isset($this->modules))
		{
			foreach ($this->modules as $id => $class)
			  Yii::import("application.modules.mongocms.modules.$id.models.*");
		}
	}


	/**
	 * AuthManager for global permissions: 'managePermissions' ...
	 *
	 * @param boolean $loadFromDB
	 * @return MongoCmsAuthManager
	 */
	public function getAuthManager($loadFromDB = true)
	{
		static $authmanager;

		if (!isset($authmanager))
			$authmanager = array();

		$authId = self::MONGOCMS_AUTHID;

		if (isset($authmanager[(integer)$loadFromDB]))
			 return $authmanager[(integer)$loadFromDB];

		$authManager = new MongoCmsAuthManager(self::MONGOCMS_AUTHID);
		if ($loadFromDB)
			$authManager->init();

		return $authmanager[(integer)$loadFromDB] = $authManager;
	}

	/**
	 * What to in mongodb if installed:
	 *
	 * - Call installContentType from every contenttype of the main mongocms module only
	 * - Call install from subModules
	 * - Install global permissisons
	 * - Install default users
	 * - Install default data
	 *
	 */
	public function install()
	{
		//only call once in the main mongocms module
		if (!$this->isSubmodule())
		{
			$this->installDefaultUsers();
			$this->installDefaultRoles();
			$this->installDefaultData();
			$this->installCMSPermissions();
		}

		//only for the contenttypes of this module
		foreach ($this->getContentTypes(false,true) as $modelClass => $label)
			Page::model($modelClass)->installContentType();

		//call install from submodules
		if (isset($this->modules))
			foreach ($this->modules as $id => $class)
			{
				$subModule = $this->getModule($id);
				if (($subModule instanceof MongoCmsModule))
					$subModule->install();
			}
	}

	/**
	 * Create default users with attributes from config/default_users.php
	 * if not exists in mongodb
	 *
	 */
	protected function installDefaultUsers()
	{
		if (User::model()->dbDataExists())
			return;

		$config = new MongoCmsConfiguration();
		$config->loadModuleConfig($this,'default_users');
		$users = $config->toArray();

		foreach ($users as $user)
	    {
	    	$model = new User;
	    	$model->setAttributes($user,false);
	    	$model->save();

			if ($model->hasErrors()) {
				echo CHtml::tag('h1',array(),__METHOD__);
				die(CHtml::errorSummary($model));
			}
	    }
	}

	/**
	 * Create default roles with attributes from config/mongocms/default_roles.php
	 * if not exists in mongodb
	 *
	 * Fieldmapping:
	 * title -> name
	 * subtitle -> description
	 *
	 */
	protected function installDefaultRoles()
	{
		if (Role::model()->dbDataExists())
			return;

		$config = new MongoCmsConfiguration();
		$config->loadModuleConfig($this,'default_roles');
		$roles = $config->toArray();

		foreach ($roles as $role => $description)
		{
			$model = new Role;
			$model->title = $role;
			$model->subtitle = $description;

			$model->save();

			if ($model->hasErrors()) {
				echo CHtml::tag('h1',array(),__METHOD__);
				die(CHtml::errorSummary($model));
			}
		}
	}


	/**
	 * Run the script config/mongocms/default_data.php
	 *
	 * @return
	 */
	protected function installDefaultData()
	{
		$config = new MongoCmsConfiguration();
		$config->includeScript('default_data');
	}


	/**
	 * Install the mongocms permissions from config/mongocms/roleoperations_mongocms.php
	 * (managePermission, ...)
	 *
	 */
	public function installCMSPermissions($recreate = false)
	{
		$authManager = $this->getAuthManager(!$recreate);
		if ($recreate || !$authManager->instanceExists())
		{
			$config = new MongoCmsConfiguration();

			$script = 'roleoperations_mongocms';
			$params = array('authManager' => $authManager);

			$config->includeScript($script,$params);
			$authManager->save();
		}
	}


	/**
	 * Check access to operations managed by global authmanager
	 *
	 * @param mixed $itemNames string or array of operations
	 * @param boolean $params the params for calculating the bizRule
	 * @param boolean $checkOnly abort on accessDenied or not
	 * @param boolean $or true: access is allowed if at least one operation of the array $itemNames is allowed,
	 *                    false: all operations have to be allowed
	 * @return boolean
	 */
	public function checkAccess($itemNames, $params = array(), $checkOnly = false, $or = true)
	{
		$authManager = $this->getAuthManager();
		return $authManager->checkUserAccess($itemNames, $params, $checkOnly, $or);
	}

	/**
	 * Returns a collection object of this db
	 *
	 * @param string $name
	 * @return mongodb collection
	 */
	public function getCollection($name) {
	  	return Yii::app()->getComponent($this->mongoConnectionId)->getDbInstance()->selectCollection($name);
	}

	/**
	 * Translation:
	 * files in in mongocms/messages
	 *
	 * @param mixed $category
	 * @param mixed $message
	 * @param integer $pluralformat
	 * @param mixed $language
	 * @return
	 */
	public static function t($message, $pluralformat = 1, $language = null, $category = 'default')
	{
		$source = self::$_translationSourceId;
		$params = array($pluralformat);
		return Yii::t($category, $message, $params, $source, $language);
	}

}

?>