<?php
/**
 * @author dongbeta
 * @copyright Copyright @ 2010 Xii Project
 * @link http://xii.googlecode.com
 * @license	 http://www.opensource.org/licenses/bsd-license.php
 * @version $Id: Limiter.php 92 2010-05-17 06:27:29Z dongbeta $
 * 
 * 行为频率控制器，可以为特定用户的特定行为设定频率限制时间。
 * 
 * 数据库格式如下：
 * CREATE TABLE `xii_limiter` (
 *   `id` int(11) unsigned NOT NULL AUTO_INCREMENT,
 *   `action` varchar(200) NOT NULL,
 *   `hash` varchar(32) NOT NULL,
 *   `create_time` int(10) unsigned NOT NULL,
 *   `period` int(10) unsigned NOT NULL,
 *   `total_times` smallint(5) unsigned NOT NULL,
 *   `remain_times` smallint(5) unsigned NOT NULL,
 *   PRIMARY KEY (`id`),
 *   UNIQUE KEY `hash` (`hash`)
 * );
 */
class Limiter extends CApplicationComponent
{
	public $table = '{{limiter}}';
	
	/**
	 * 检查限制器，如果没有则建立
	 * @param string $name 限制器的名称
	 * @param mixed $params 用来定位控制器的参数
	 * @param int $period 控制时间
	 * @param int $times 在控制时间内的可操作次数
	 * @return boolean
	 */
	public function check($action, $params, $period, $total_times)
	{
		$hash = $this->hash($action, $params);
		if(!($limit = $this->get($action, $hash)))
			return $this->create($action, $hash, $period, $total_times);
		
		if($this->needUpdate($limit, $period, $total_times))
			$this->update($limit, $period, $total_times);
		
		return $this->validate($limit);
	}

	public function spend($action, $params)
	{
		$hash = $this->hash($action, $params);
		$limit = $this->get($action, $hash);
		if(!$limit instanceof LimiterRecord)
			return false;
		$limit->remain_times = $limit->remain_times - 1;
		return $limit->save();
	}

	/**
	 * hash the params
	 * @param string $action
	 * @param mixed $params
	 * @return string
	 */
	protected function hash($action, $params)
	{
		return md5(serialize($params) . $action);
	}

	/**
	 * get item form date base
	 * @param string $action
	 * @param string $hash
	 * @return object LimiterRecord
	 */
	protected function get($action, $hash)
	{
		$criteria = new CDbCriteria();
		$criteria->condition = 'action=:action AND hash=:hash';
		$criteria->params = array(':action'=>$action, 'hash'=>$hash);
		return LimiterRecord::model()->find($criteria);
	}

	/**
	 * create item in datebase
	 * @param string $action
	 * @param string $hash
	 * @param int $period
	 * @param int $total_times
	 * @return boolean
	 */
	protected function create($action, $hash, $period, $total_times)
	{
		$limit = new LimiterRecord();
		$limit->action = $action;
		$limit->hash = $hash;
		$limit->period = $period;
		$limit->create_time = time();
		$limit->total_times = $total_times;
		$limit->remain_times = $total_times;
		return $limit->save();
	}

	/**
	 * check if a limiter need be update
	 * @param boject $limit LimiterRecord
	 * @param int $period
	 * @param int $total_times
	 * @return boolean
	 */
	protected function needUpdate($limit, $period, $total_times)
	{
		return $limit->period != $period || $limit->total_times != $total_times || $limit->create_time + $limit->period < time();
	}

	/**
	 * update information of one limiter item
	 * @param int $limit
	 * @param int $period
	 * @param int $total_times
	 * @return boolean
	 */
	protected function update($limit, $period, $total_times)
	{
		$limit->period = $period;
		$limit->total_times = $total_times;
		$limit->remain_times = $total_times;
		$limit->create_time = time();
		return $limit->save();
	}

	/**
	 * validate a limit
	 * @param boject $limit Limit
	 * @return boolean
	 */
	protected function validate($limit)
	{
		return $limit->remain_times > 0;
	}
}
/**
 * 限制器的数据库模型
 */
class LimiterRecord extends CActiveRecord
{

	public static function model()
	{
		return parent::model(__CLASS__);
	}

	public function tableName()
	{
		return Yii::app()->limiter->table;
	}

	public function rules()
	{
		return array(
			array('action', 'match', 'pattern'=>'/^([a-zA-Z0-9]+\.)*[a-zA-Z0-9]+$/'), 
			array('hash', 'match', 'pattern'=>'/^[a-z0-9]{32}$/'), 
			array('create_time', 'match', 'pattern'=>'/^[0-9]+$/'), 
			array('period', 'match', 'pattern'=>'/^[0-9]+$/'), 
			array('total_times', 'match', 'pattern'=>'/^[0-9]+$/'), 
			array('remain_times', 'match', 'pattern'=>'/^[0-9]+$/')
		);
	}
}