<?php
/**
 * JTaconite class file.
 *
 * @author Stefan Volkmar <volkmar_yii@email.de>
 * @license BSD
 * @version 1.2
 */

/**
 * This widget encapsulates the Taconite jQuery plugin.
 * The plugin is a port of Taconite to jQuery. Make multiple DOM manipulations in 
 * response to a single remote call using an XML command document
 * ({@link http://jquery.malsup.com/taconite/}).
 *
 * @author Stefan Volkmar <volkmar_yii@email.de>
 */

class JTaconite extends CWidget
{
	/**
	 * @var boolean set to true to enable debug logging to window.console.log
	 * Defaults to 0.
	 */
	public $debug;
	/**
	 * @var boolean enable the detection of the AJAX response content-type
	 * Defaults to true.
	 */
	public $autodetect;
	/**
	 * @var array other default setting for the plugin
     *
	 */
	public $defaults=false;

	/**
	 * @var array internal List for javascript code pieces
	 */
    protected $scriptLines=array();
	/**
	 * @var object Client script
	 */
    protected $cs;
	
	/**
	 * Initializes the widget.
	 * This method registers all needed client scripts 
	 */
	public function init()
	{
		$taconiteFile=(YII_DEBUG)
				?'/js/jquery.taconite.js'
				:'/js/jquery.taconite.min.js';
				
  		$this->cs = Yii::app()->getClientScript()
            ->registerCoreScript('jquery')
            ->registerScriptFile(CHtml::asset(dirname(__FILE__).DIRECTORY_SEPARATOR.'assets').$taconiteFile);

        $js = "";
        $js .= ($this->debug===true)
             ?"$.taconite.debug = true;\n"
             :'';

        $js .= ($this->autodetect===false)
             ?"$.taconite.autodetect = false;\n"
             :'';

        $js .= (is_array($this->defaults))
             ?"$.taconite.defaults = ".CJavaScript::encode($this->defaults).";"
             :'';

        $id = $this->getId();
        if ($js)
            $this->cs->registerScript(__CLASS__.'#head#'.$id, $js, CClientScript::POS_HEAD);
	}

	/**
	 * Add code pieces of Javascript to the internal list.
     * @param array $code Javascript code
	 */
    public function addScriptLines(array $code)
    {
        $this->scriptLines=array_merge($this->scriptLines, $code);

    }

	/**
	 * Clear the internal list for Javascript.
	 */
    private function resetScriptLines()
    {
        $this->scriptLines=array();
    }

	/**
	 * Generates a piece of Javascript - event handler for a HTML element.
     * @param string $selector the jQuery selector
     * @param string $eventName the name of a javascript event
     * @param array $functionBody the code lines
     * @return void
	 */
    public function addEventHandler($selector, $eventName, array $functionBody){
        $code=array();
        $code[] = "$('$selector').$eventName(function(){";
        $code[] = implode("\n", $functionBody);
        $code[] = "});";
        $this->addScriptLines($code);
    }

    /**
     * Registers and generate the javascript code for the plugin.
     * @param integer $position the position of the JavaScript code. Valid values include the following:
     * <ul>
     * <li>CClientScript::POS_HEAD : the script is inserted in the head section right before the title element.</li>
     * <li>CClientScript::POS_BEGIN : the script is inserted at the beginning of the body section.</li>
     * <li>CClientScript::POS_END : the script is inserted at the end of the body section.</li>
     * <li>CClientScript::POS_LOAD : the script is inserted in the window.onload() function.</li>
     * <li>CClientScript::POS_READY : the script is inserted in the jQuery's ready function.</li>
     * </ul>
     */
    public function registerScript($position=CClientScript::POS_READY){
        $js = implode("\n", $this->scriptLines);		
        $this->cs->registerScript(__CLASS__.'#'.$this->getId().$position, $js, $position);
        $this->resetScriptLines();
    }

    /**
     * Generates the JavaScript that initiates an AJAX request.
     * @param array $options AJAX options. The valid options are specified in the jQuery ajax documentation.
     * The following special options are added for convenience:
     * <ul>
     * <li>update: string, specifies the selector whose HTML content should be replaced
     *   by the AJAX request result.</li>
     * <li>replace: string, specifies the selector whose target should be replaced
     *   by the AJAX request result.</li>
     * </ul>
     * Note, if you specify the 'success' option, the above options will be ignored.
     * @return string the generated JavaScript
     * @see http://docs.jquery.com/Ajax/jQuery.ajax#options
     */
    public static function ajax($options){
        return CHtml::ajax($options);
    }

    /**
     * Generates the JavaScript that load data from the server using a HTTP GET request.
     * @param string $url the URL to which the request is sent
     * @param array $options AJAX options. The valid options are specified in the jQuery ajax documentation.
     * The following special options are added for convenience:
     * <ul>
     * <li>update: string, specifies the selector whose HTML content should be replaced
     *   by the AJAX request result.</li>
     * <li>replace: string, specifies the selector whose target should be replaced
     *   by the AJAX request result.</li>
     * </ul>
     * Note, if you specify the 'success' option, the above options will be ignored.
     * @return string the generated JavaScript
     * @see http://api.jquery.com/jQuery.get/
     */

    public static function get($url, $options=array()){
        $options['type'] = 'get';
        $options['url']  = $url;
        return CHtml::ajax($options);
    }
    /**
     * Generates the JavaScript that load data from the server using a HTTP POST request.
     * @param string $url the URL to which the request is sent
     * @param array $options AJAX options. The valid options are specified in the jQuery ajax documentation.
     * The following special options are added for convenience:
     * <ul>
     * <li>update: string, specifies the selector whose HTML content should be replaced
     *   by the AJAX request result.</li>
     * <li>replace: string, specifies the selector whose target should be replaced
     *   by the AJAX request result.</li>
     * </ul>
     * Note, if you specify the 'success' option, the above options will be ignored.
     * @return string the generated JavaScript
     * @see http://api.jquery.com/jQuery.post/
     */
    public static function post($url, $options=array()){
        $options['type'] = 'post';
        $options['url']  = $url;
        return CHtml::ajax($options);
    }
}
