<?php
/**
 * EEmailSmtpValidator class file.
 *
 * @author MetaYii
 * @link http://www.yiiframework.com/
 * @copyright Copyright &copy; 2008 MetaYii
 * @version 1.0
 * @license
 *
 * Copyright © 2008 by MetaYii
 * All rights reserved.
 *
 * This software is released under the Creative Commons Attribution license 3.0, which
 * can be found here:
 *
 * @link http://creativecommons.org/licenses/by/3.0/legalcode
 *
 * and gives you these freedoms:
 *
 * @link http://creativecommons.org/licenses/by/3.0/
 *
 * DISCLAIMER:
 *
 * UNLESS OTHERWISE MUTUALLY AGREED TO BY THE PARTIES IN WRITING, LICENSOR OFFERS THE WORK
 * AS-IS AND MAKES NO REPRESENTATIONS OR WARRANTIES OF ANY KIND CONCERNING THE WORK, EXPRESS,
 * IMPLIED, STATUTORY OR OTHERWISE, INCLUDING, WITHOUT LIMITATION, WARRANTIES OF TITLE,
 * MERCHANTIBILITY, FITNESS FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF
 * LATENT OR OTHER DEFECTS, ACCURACY, OR THE PRESENCE OF ABSENCE OF ERRORS, WHETHER OR NOT
 * DISCOVERABLE. SOME JURISDICTIONS DO NOT ALLOW THE EXCLUSION OF IMPLIED WARRANTIES, SO
 * SUCH EXCLUSION MAY NOT APPLY TO YOU.
 *
 * EXCEPT TO THE EXTENT REQUIRED BY APPLICABLE LAW, IN NO EVENT WILL LICENSOR BE LIABLE
 * TO YOU ON ANY LEGAL THEORY FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL, PUNITIVE OR
 * EXEMPLARY DAMAGES ARISING OUT OF THIS LICENSE OR THE USE OF THE WORK, EVEN IF LICENSOR
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * --------------------------------------------------------------------------------------
 * This work was inspired by the SMTP_validateEmail class written by Gabe.
 * @link http://code.google.com/p/php-smtp-email-validation/
 * The copyright for that class follows:
 * Validate Email Addresses Via SMTP
 * This queries the SMTP server to see if the email address is accepted.
 * @copyright http://creativecommons.org/licenses/by/3.0/ - Please keep this comment intact
 * @author gabe@fijiwebdesign.com
 * @contributers adnan@barakatdesigns.net
 * @version 0.1a
 * --------------------------------------------------------------------------------------
 */

/**
 * EEmailSmtpValidator validates that the attribute value is a valid e-mail address,
 * using SMTP.
 *
 * @author MetaYii
 * @package application.extensions.emailsmtpvalidator
 * @since 1.0
 * @uses Pear's Net/DNS on Windows(tm)
 */
class EEmailSmtpValidator extends CValidator
{
   // *************************
   // Properties:
   // *************************

   /**
    * SMTP Port
    */
   private $port = 25;
   /**
    * Nameservers to use when make DNS query for MX entries
    * @var Array $nameservers
    */
   private $nameServers = array('localhost');
   /**
    * How many seconds to wait before each attempt to connect to the
    * destination e-mail server
    *
    * @var integer
    */
   private $timeOut = 10;
   /**
    * How many seconds to wait for data exchanged with the server.
    * Set to a non zero value if the data timeout will be different
    * than the connection timeout.
    *
    * @var integer
    */
   private $dataTimeOut = 5;
   /**
    * The address of the sending user
    *
    * @var string
    */
   private $sender = 'postmaster@localhost';
   /**
    * If it is not possible to verify if the e-mail address is valid,
    * and this flag is set to true, then the validation will fail.
    *
    * @var boolean
    */
   private $strictValidation = false;

   // *************************
   // Private properties:
   // *************************

   /**
    * PHP Socket resource to remote MTA. This is just a private property.
    * @var resource $sock
    */
   private $sock;

   // *************************
   // Setters and getters:
   // *************************

   /**
    * Set the SMTP port
    *
    * @param integer $value
    */
   public function setPort($value)
   {
      if (!is_integer($port))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->port = $value;
   }

   /**
    * Returns the SMTP port
    *
    * @return integer
    */
   public function getPort()
   {
      return $this->port;
   }

   /**
    * Sets the array of nameservers
    *
    * @param array $value
    */
   public function setNameServers($value)
   {
      if (!is_array($value) || empty($value))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->nameServers = $value;
   }

   /**
    * Returns the array of nameservers
    *
    * @return array
    */
   public function getNameServers()
   {
      return $this->nameServers;
   }

   /**
    * Sets the socket timeout
    *
    * @param integer $value
    */
   public function setTimeOut($value)
   {
      if (!is_integer($value))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->timeOut = abs($value);
   }

   /**
    * Returns the socket timeout
    *
    * @return integer
    */
   public function getTimeOut()
   {
      return $this->timeOut;
   }

   /**
    * Sets the data timeout
    *
    * @param integer $value
    */
   public function setDataTimeOut($value)
   {
      if (!is_integer($value))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->dataTimeOut = abs($value);
   }

   /**
    * Sets the sender
    *
    * @param string $value
    */
   public function setSender($value)
   {
      if (!is_string($value))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->sender = $value;
   }

   /**
    * Returns the sender
    *
    * @return string
    */
   public function getSender()
   {
      return $this->sender;
   }

   /**
    * Sets if the validation should be strict
    *
    * @param boolean $value
    */
   public function setStrictValidation($value)
   {
      if (!is_bool($value))
         throw new CException('EEmailSmtpValidator', 'Invalid value.');
      $this->strictValidation = $value;
   }

   /**
    * Returns if the validation should be strict
    *
    * @return boolean
    */
   public function getStrictValidation()
   {
      return $this->strictValidation;
   }

   // *************************
   // Private methods:
   // *************************

   private function parseEmail($email)
   {
      $parts = explode('@', $email);
      $domain = array_pop($parts);
      $user= implode('@', $parts);
      return array($user, $domain);
   }

   private function send($msg)
   {
      fwrite($this->sock, $msg."\r\n");
      $reply = fread($this->sock, 2082);
      return $reply;
   }

   /**
    * Query DNS server for MX entries
    * @return
    */
   private function queryMX($domain)
   {
      $hosts = array();
      $mxweights = array();
      if (function_exists('getmxrr')) {
         getmxrr($domain, $hosts, $mxweights);
      }
      else {
         set_include_path(get_include_path().PATH_SEPARATOR.dirname(__FILE__));
         require_once(dirname(__FILE__).DIRECTORY_SEPARATOR.'Net'.DIRECTORY_SEPARATOR.'DNS.php');
         $resolver = new Net_DNS_Resolver();
         $resolver->nameservers = $this->nameServers;
         $resp = $resolver->query($domain, 'MX');
         if ($resp) {
            foreach ($resp->answer as $answer) {
               $hosts[] = $answer->exchange;
               $mxweights[] = $answer->preference;
            }
         }
      }
      return array($hosts, $mxweights);
   }

   // *************************
   // Validator method:
   // *************************

	/**
	 * Validates the attribute of the object.
	 * If there is any error, the error message is added to the object.
	 * @param CModel the object being validated
	 * @param string the attribute being validated
	 */
	protected function validateAttribute($object, $attribute)
	{
	   $valid = false;

      if (is_object($object) && isset($object->$attribute)) $email = $object->$attribute;

      if (isset($email) && strcmp($email, '')) {
         list($localUser, $localHost) = $this->parseEmail($this->sender);
         list($user, $domain) = $this->parseEmail($email);

         $mxs = array();
         list($hosts, $mxweights) = $this->queryMX($domain);

         $c = count($hosts);
         for ($i=0; $i<$c; $i++) {
            $mxs[$hosts[$i]] = $mxweights[$i];
         }
         asort($mxs);

         $to = $this->timeOut;
         while (list($host) = each($mxs)) {
            if ($this->sock = @fsockopen($host, $this->port, $errno, $errstr, (float)$to)) {
               stream_set_timeout($this->sock, $this->dataTimeOut);
               break;
            }
         }

         if ($this->sock) {
            $reply = fread($this->sock, 2082);

            preg_match('/^([0-9]{3}) /ims', $reply, $matches);
            $code = isset($matches[1]) ? $matches[1] : '';

            if ($code != '220') {
               $valid = false;
            }
            else {
               $this->send("HELO ".$localHost);
               $this->send("MAIL FROM: <".$localUser.'@'.$localHost.">");
               $reply = $this->send("RCPT TO: <".$user.'@'.$domain.">");
               preg_match('/^([0-9]{3}) /ims', $reply, $matches);
               $code = isset($matches[1]) ? $matches[1] : '';
               if ($code == '250') {
                  $valid = true;
               }
               elseif ($code == '451' || $code == '452') {
                  $valid = $this->strictValidation ? false : true;
               }
               else {
                  $valid = false;
               }
               $this->send("RSET");
               $this->send("quit");
               fclose($this->sock);
            }
         }
         else {
   		   $valid = false;
         }
      }

		if (!$valid) {
		   $message = $this->message !== null ? $this->message : Yii::t('EEmailValidator', 'The e-mail address is invalid.');
			$this->addError($object, $attribute, $message);
		}
	}
}