<?php

/**
 * EDbStatePersister implements IStatePersister using DB as storage
 *
 * @author Pavel Voronin <pavel.a.voronin@gmail.com>
 * @license MIT
 * @version 1.0
 */

/**
* The MIT License (MIT)
* 
* Copyright (c) 2013 Pavel Voronin
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy of
* this software and associated documentation files (the "Software"), to deal in
* the Software without restriction, including without limitation the rights to
* use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
* the Software, and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
* COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
* IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

/**
 * EDbStatePersister class
 */
class EDbStatePersister extends CApplicationComponent implements IStatePersister
{
	/**
	 * @var string the ID of the db connection
	 */
	public $dbConnection = 'db';

	/**
	 * @var CDbConnection DB connection object
	 */
	protected $db;

	/**
	 * @var string the table name storing the state data
	 */
	public $tableName = 'states';

	/**
	 * Initializes the component.
	 * This method overrides the parent implementation by making sure {@link tableName}
	 * contains valid value.
	 * @return void
	 */
	public function init ( )
	{
		parent::init ( );

		$this->db = $db = Yii::app()->getComponent ( $this->dbConnection );

		if ( $db === null )
			throw new CException ( Yii::t ( 'yii', '\'{dbConnection}\' component doesn\'t exist.', array ( '{dbConnection}' => $this->dbConnection ) ) );

		if ( ! ( $db instanceof CDbConnection ) )
			throw new CException ( Yii::t ( 'yii', '\'{dbConnection}\' component is not correct CDbConnection descendant.', array ( '{dbConnection}' => $this->dbConnection ) ) );

		if ( $this->tableName === null )
			throw new CException ( Yii::t ( 'yii', 'tableName param cannot be null.' ) );

		if ( $db->schema->getTable ( $this->tableName ) === null )
			$this->createTable ( );
	}

	/**
	 * Loads state data from db persistent storage
	 * @return mixed state data. Null if no state data available.
	 */
	public function load ( )
	{
		$state = $this->db->createCommand ( )->select ( 'value' )->from ( $this->tableName )->where ( '`key` = :key', array ( ':key' => Yii::app()->name ) )->queryScalar ( ) ;

		if ( $state !== false )
			return unserialize ( $state );
		else
			return null;
	}

	/**
	 * Saves application state in persistent storage.
	 * @param mixed $state state data (must be serializable).
	 */
	public function save ( $state )
	{
		$oldState = $this->db->createCommand ( )->select ( 'value' )->from ( $this->tableName )->where ( '`key` = :key', array ( ':key' => Yii::app()->name ) )->queryScalar ( );

		if ( $oldState === false )
			$this->db->createCommand ( )->insert ( $this->tableName, array ( 'key' => Yii::app()->name, 'value' => serialize ( $state ) ) );
		else
			$this->db->createCommand ( )->update ( $this->tableName, array ( 'value' => serialize ( $state ) ), '`key` = :key', array ( 'key' => Yii::app()->name ) );
	}

	/**
	 * Clears application state in persistent storage.
	 * @param mixed $state state data (must be serializable).
	 */
	public function clear ( )
	{
		$this->db->createCommand ( )->delete ( $this->tableName, '`key` = :key', array ( 'key' => Yii::app()->name ) );
	}

	/**
	 * Creates table with tableName
	 */
	protected function createTable ( )
	{
		try
		{
			$command = $this->db->createCommand ( );
			$command->createTable ( $this->tableName, array ( 'key' => 'string NOT NULL', 'value' => 'text NOT NULL', 'PRIMARY KEY (`key`)' ) );
		}
		catch ( CDbException $e )
		{
			throw new CException ( Yii::t ( 'yii', 'Can\'t create state persister table because of lack of access. Please, grant CREATE privilege to \'{dbConnection}\' connection user or create table manually with SQL: {sql}.', array ( '{dbConnection}' => $this->dbConnection, '{sql}' => $command->text ) ) );
		}
	}
}
